package com.tykj.model_layer.service.impl;

import com.github.wenhao.jpa.PredicateBuilder;
import com.github.wenhao.jpa.Specifications;

import com.tykj.base.result.ApiException;
import com.tykj.model_layer.dao.ColumnInfoDao;
import com.tykj.model_layer.dao.QuoteDao;
import com.tykj.model_layer.dao.TableInfoDao;
import com.tykj.model_layer.entity.ColumnInfo;
import com.tykj.model_layer.entity.Quote;
import com.tykj.model_layer.entity.TableInfo;
import com.tykj.model_layer.entity.customEnums.ConnectionType;
import com.tykj.model_layer.entity.vo.*;
import com.tykj.model_layer.myEnum.ModelType;
import com.tykj.model_layer.service.ModelService;
import com.tykj.model_layer.utils.SessionUtil;
import com.tykj.model_layer.utils.SqlUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import org.apache.logging.log4j.util.Strings;
import org.hibernate.Session;
import org.hibernate.StaleStateException;
import org.hibernate.internal.SessionImpl;
import org.hibernate.persister.entity.EntityPersister;
import org.hibernate.query.Query;
import org.hibernate.type.IntegerType;
import org.hibernate.type.StringType;
import org.hibernate.type.TimestampType;
import org.hibernate.type.Type;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.data.domain.Page;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;


import javax.persistence.criteria.Path;
import javax.persistence.criteria.Predicate;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;


import static com.tykj.model_layer.utils.CreateTableUtil.createTable;
import static com.tykj.model_layer.utils.CreateTableUtil.createTableBack;
import static com.tykj.model_layer.utils.HqlUtil.createQuery;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;


/**
 * @ClassName ModelImpl
 * @Description TODO
 * @Author WWW
 * @Date 2021/2/26 13:39
 * @Version 1.0
 */

@Service
@Slf4j
public class ModelImpl implements ModelService {


    @Autowired
    private TableInfoDao tableInfoDao;

    @Autowired
    private ColumnInfoDao columnInfoDao;

    @Autowired
    private QuoteDao quoteDao;

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private SessionUtil sessionUtil;

    @Autowired
    private Environment env;

    /**
     * @param
     * @return java.util.List<com.tykj.workflowcore.model_layer.model.TableInfo>
     * @Author WWW
     * @Description 得到所有表
     * @Date 16:14 2021/3/5
     **/
    @Override
    public Page<TableInfo> pageAllEntities(SearchTableInfoVo searchTableInfoVo) {
        PredicateBuilder<TableInfo> and = createPredicateBySearchTableInfoVo(searchTableInfoVo);
        return tableInfoDao.findAll(and.build(), searchTableInfoVo.getPageable());

    }

    @Override
    public List<TableInfo> listAllEntities(SearchTableInfoVo searchTableInfoVo) {
        PredicateBuilder<TableInfo> and = createPredicateBySearchTableInfoVo(searchTableInfoVo);
        return tableInfoDao.findAll(and.build());
    }

    private PredicateBuilder<TableInfo> createPredicateBySearchTableInfoVo(SearchTableInfoVo searchTableInfoVo) {
        PredicateBuilder<TableInfo> and = Specifications.and();
        if (searchTableInfoVo != null) {
            and.like(searchTableInfoVo.getModelName() != null && StringUtils.isNotEmpty(searchTableInfoVo.getModelName()), "modelName", "%" + searchTableInfoVo.getModelName() + "%");
            and.like(searchTableInfoVo.getModelTitle() != null && StringUtils.isNotEmpty(searchTableInfoVo.getModelTitle()), "modelTitle", "%" + searchTableInfoVo.getModelTitle() + "%");
            and.in(searchTableInfoVo.getModelType() != null && searchTableInfoVo.getModelType().length > 0, "modelType", searchTableInfoVo.getModelType());
            and.in(searchTableInfoVo.getIds() != null && searchTableInfoVo.getIds().length > 0, "id", searchTableInfoVo.getIds());
        }
        return and;
    }

    /**
     * @param searchColumnInfoVo
     * @return java.util.List<com.tykj.workflowcore.model_layer.model.ColumnInfo>
     * @Author WWW
     * @Description 根据表名得到所有字段名
     * @Date 16:15 2021/3/5
     **/
    @Override
    public List<ColumnInfo> showModelFields(SearchColumnInfoVo searchColumnInfoVo) {
        PredicateBuilder<ColumnInfo> and = Specifications.and();
        and.eq(searchColumnInfoVo.getDbId() != null, "dbId", searchColumnInfoVo.getDbId());
        and.eq(searchColumnInfoVo.getDbName() != null && StringUtils.isNotEmpty(searchColumnInfoVo.getDbName()),
                "dbName", searchColumnInfoVo.getDbName());
        return columnInfoDao.findAll(and.build());
    }


    /**
     * @param tableVO
     * @return com.tykj.workflowcore.model_layer.model.TableVO
     * @Author WWW
     * @Description 完全新建一张表
     * @Date 16:16 2021/3/5
     **/
    @Override
    public TableInfo newTable(TableVO tableVO) {
        String xmlMapping = createTable(tableVO);
        String xmlBack = createTableBack(tableVO);
        sessionUtil.addXml(xmlMapping);
        sessionUtil.addXml(xmlBack);
        Session session = sessionUtil.getSession();
        List<ColumnVO> dataList = tableVO.getDataList();
        List<String> collect = tableInfoDao.findAll()
                .stream()
                .map(TableInfo::getModelName)
                .collect(Collectors.toList());
        if (collect.contains(tableVO.getModelName())) {
            throw new ApiException("表名已经存在！");
        }
        TableInfo tableInfo = new TableInfo();
        tableInfo.setModelName(tableVO.getModelName());
        tableInfo.setModelTitle(tableVO.getModelTitle());
        tableInfo.setXml(xmlMapping);
        tableInfo.setBackName(tableVO.getModelName() + "_back");
        tableInfo.setBackXml(xmlBack);
        tableInfo.setModelType(tableVO.getModelType());
        tableInfo.setDescription(tableVO.getDescription());
        tableInfoDao.save(tableInfo);
        //默认存储ID字段
        if (tableInfo.getModelType().equals(ModelType.VIRTUAL)) {
            ColumnInfo cId = new ColumnInfo(0, "id", "主键", "java.lang.Integer", 11, tableInfo.getModelName(), tableInfo.getId(), "主键");
            columnInfoDao.save(cId);
        }
        for (ColumnVO columnVO : dataList) {
            ColumnInfo columnInfo = new ColumnInfo();
            columnInfo.setFieldName(columnVO.getFieldName());
            columnInfo.setFieldType(columnVO.getFieldType());
            columnInfo.setFieldLength(columnVO.getFieldLength());
            columnInfo.setFieldTitle(columnVO.getFieldTitle());
            columnInfo.setDescription(columnVO.getDescription());
            columnInfo.setDbName(tableInfo.getModelName());
            columnInfo.setDbId(tableInfo.getId());
            columnInfo.setPrimaryKey(columnVO.getPrimaryKey());
            ColumnInfo save = columnInfoDao.save(columnInfo);
            List<Quote> quoteList = columnVO.getQuoteList();
            if (quoteList != null) {
                for (Quote quote : quoteList) {
                    if (StringUtils.isNotEmpty(quote.getValue())) {
                        quote.setColumnId(save.getId());
                        quoteDao.save(quote);
                    }
                }
            }
        }
        //关闭会话
        session.close();
        return tableInfo;
    }

    /**
     * @param map
     * @return int
     * @Author WWW
     * @Description 根据表名新增数据
     * @Date 16:17 2021/3/5
     **/
    @Override
    public int putValueByEntityName(Map<String, Object> map, Boolean isDelete) {
        for (String tableName :
                map.keySet()) {
            //查找对应的表
            Specification spec = (root, criteriaQuery, criteriaBuilder) -> {
                Predicate equal;
                Path name = root.get("modelName");
                equal = criteriaBuilder.equal(name, tableName);
                return equal;
            };
            Optional one = tableInfoDao.findOne(spec);
            TableInfo tableInfo = null;
            if (one.isPresent()) {
                tableInfo = (TableInfo) one.get();
            }
            SessionImpl session = (SessionImpl) sessionUtil.getSession();
            session.getTransaction().begin();
            Object values = map.get(tableName);
            if (values instanceof Map) {
                //插入数据
                insertValue(tableInfo.getModelName(), (Map) values, session, true, false, isDelete);

            } else {
                //循环插入数据
                List valuesList = (List) values;
                for (int i = 0; i < valuesList.size(); i++) {
                    insertValue(tableInfo.getModelName(), (Map) valuesList.get(i), session, true, false, isDelete);
                }
            }
            try {
                session.getTransaction().commit();
            } catch (Exception e) {
                //todo 等会捕捉一下
                if (e.getMessage().contains("Batch update returned unexpected row count from")) {
                    throw new ApiException("要操作对象的主键不存在");
                }
            } finally {
                session.close();
            }


        }
        return 0;
    }

    /**
     * @param
     * @return int
     * @Author WWW
     * @Description 根据表名新增数据
     * @Date 16:17 2021/3/5
     **/
    @Override
    public int putValueByEntityNameList(List<Map<String, Object>> mapList, Boolean isDelete) {
        SessionImpl session = (SessionImpl) sessionUtil.getSession();
        session.getTransaction().begin();
        for (Map<String, Object> map : mapList) {
            for (String tableName :
                    map.keySet()) {
                //查找对应的表
                Specification spec = (root, criteriaQuery, criteriaBuilder) -> {
                    Predicate equal;
                    Path name = root.get("modelName");
                    equal = criteriaBuilder.equal(name, tableName);
                    return equal;
                };
                Optional one = tableInfoDao.findOne(spec);
                TableInfo tableInfo = null;
                if (one.isPresent()) {
                    tableInfo = (TableInfo) one.get();
                }
                Object values = map.get(tableName);
                if (values instanceof Map) {
                    //插入数据
                    insertValue(tableInfo.getModelName(), (Map) values, session, true, false, isDelete);
                } else {
                    //循环插入数据
                    List valuesList = (List) values;
                    for (int i = 0; i < valuesList.size(); i++) {
                        insertValue(tableInfo.getModelName(), (Map) valuesList.get(i), session, true, false, isDelete);
                    }
                }

            }
        }
        try {
            session.getTransaction().commit();
        } catch (StaleStateException staleStateException) {
            if (staleStateException.getMessage().contains("Batch update returned unexpected row count from")) {
                throw new ApiException("要操作对象的主键不存在");
            }
        } finally {
            session.close();
        }
        return 0;
    }


    /**
     * @param tableName
     * @param map
     * @return void
     * @Author WWW
     * @Description 新增参数的方法
     * @Date 16:17 2021/3/5
     **/
    public void insertValue(String tableName, Map map, SessionImpl session, boolean back, boolean saveOrSaveAndUpdate, boolean isDelete) {
        EntityPersister entityPersister = session.getEntityPersister(tableName, map);
        //主键类型推断
        String identifierPropertyName = entityPersister.getIdentifierPropertyName();
        if (map.containsKey(identifierPropertyName)) {
            changeValueToTargetType(map, map.get(identifierPropertyName), entityPersister.getIdentifierType(), identifierPropertyName);
        }
        //类型推断 根据目标值的类型强转目标值的目标类型
        Type[] propertyTypes = entityPersister.getPropertyTypes();
        String[] propertyNames = entityPersister.getEntityPersister().getPropertyNames();
        Object[] propertyValuesToInsert = entityPersister.getPropertyValuesToInsert(map, null, session);
        for (int i = 0; i < propertyValuesToInsert.length; i++) {
            Object value = propertyValuesToInsert[i];
            Type propertyType = propertyTypes[i];
            //根据目标Type转换
            changeValueToTargetType(map, value, propertyType, propertyNames[i]);
        }
        HashMap hashMap = new HashMap();
        hashMap.putAll(map);
        //到这里直接执行删除操作
        if (isDelete) {
            session.delete(tableName, map);
            return;
        }
        if (saveOrSaveAndUpdate) {
            session.save(tableName, map);
        } else {
            session.saveOrUpdate(tableName, map);
        }

        if (back) {
            if (saveOrSaveAndUpdate) {
                session.save(tableName + "_back", map);
            } else {
                session.saveOrUpdate(tableName + "_back", map);
            }
        }
    }


    /**
     * @param tableName
     * @return boolean
     * @Author WWW
     * @Description 判重
     * @Date 10:50 2021/3/11
     **/
    public TableInfo getTableInfoByTableName(String tableName) {
        Specification spec = (root, criteriaQuery, criteriaBuilder) -> {
            Path name = root.get("modelName");
            Predicate equal = criteriaBuilder.equal(name, tableName);
            return equal;
        };
        List<TableInfo> all = tableInfoDao.findAll(spec);

        if (all != null && all.size() > 0) {
            return all.get(0);
        }
        return null;
    }

    /**
     * @param name
     * @return java.util.List
     * @Author WWW
     * @Description 通过表名查询所有信息
     * @Date 10:51 2021/3/11
     **/
    @Override
    public List<Map<String, Object>> findAllByName(String name) {
        if (name != null && name != "") {
            String sql = "select * from " + name;
            return jdbcTemplate.queryForList(sql);
        }
        return null;
    }

    @Override
    public List<Map<String, Object>> complexQuery(String tableName, List<String> columnNames, List<QueryCondition> queryConditions, String groupName) {
        if (Strings.isNotEmpty(tableName)) {
            String query = createQuery(tableName, columnNames, queryConditions, groupName);
            Session session = sessionUtil.getSession();
            Query query1 = session.createQuery(query);
            List<Map<String, Object>> list = query1.list();
            session.close();
            return list;
        }
        return null;
    }

    @Override
    public CustomPage complexQueryWithGroup(String tableName, GroupCondition groupCondition, List<String> columnNames, List<QueryCondition> queryConditions, String groupByColumn, Integer page, Integer size) {
        if (Strings.isNotEmpty(tableName)) {
            List<QueryCondition> queryConditionList = new ArrayList<>();
            if (queryConditions != null) {
                queryConditionList.addAll(queryConditions);
            }
            if (groupCondition != null) {
                if (StringUtils.isNotEmpty(groupCondition.getValue())) {
                    queryConditionList.add(new QueryCondition(
                            groupByColumn,
                            "like",
                            groupCondition.getValue(),
                            ConnectionType.AND
                    ));
                }
            }
            String query = createQuery(tableName, columnNames, queryConditionList, Strings.EMPTY);
            Session session = sessionUtil.getSession();
            Query query1 = session.createQuery(query);
            List list1 = query1.list();
            page = page - 1;
            query1.setFirstResult(page * size);
            query1.setMaxResults(size);
            List list2 = query1.list();
            session.close();
            return new CustomPage(list1.size(), list2);
        }


//        if (Strings.isNotEmpty(tableName)) {


//            }else {
//                for (GroupCondition groupCondition : groupConditions) {
//                    List<QueryCondition> queryConditionList = new ArrayList<>();
//                    if (queryConditions!=null){
//                        queryConditionList.addAll(queryConditions);
//                    }
//                    if (groupByColumn.isEmpty()){
//                        throw new ApiException("分类字段不能为空");
//                    }
//                    queryConditionList.add(new QueryCondition(
//                            groupByColumn,
//                            "like",
//                            groupCondition.getValue(),
//                            ConnectionType.AND
//                    ));
//                    String query = createQuery(tableName, columnNames, queryConditionList, Strings.EMPTY);
//                    Session session = sessionUtil.getSession();
//                    Query query1 = session.createQuery(query);
//                    List<Map<String, Object>> list = query1.list();
//                    result.put(groupCondition.getName(), list);
//                    session.close();
//                }
//            }
//            return result;
//        }
        return null;
    }

    private QueryCondition queryCondition(String groupByColumn, GroupCondition groupCondition) {
        return new QueryCondition(
                groupByColumn,
                "like",
                groupCondition.getValue(),
                ConnectionType.AND
        );
    }

    @Override
    public int updateTable(UpdateTableInfoVO updateTableInfoVO) {
        //tableInfo和columnInfo变化
        //查询到TableInfo和ColumnInfo
        TableVO tableVO = updateTableInfoVO.getTableVO();
        Integer dbId = updateTableInfoVO.getDbId();
        TableInfo tableInfo = tableInfoDao.findById(dbId).orElseThrow(() -> new RuntimeException("未找到该id的表信息"));
        tableInfo.setUpdatedTime(new Date());
        tableInfo.setDescription(tableVO.getDescription());
        tableInfo.setModelTitle(tableVO.getModelTitle());
        tableInfo.setModelType(tableVO.getModelType());
        String xml = createTable(tableVO);
        String backXml = createTableBack(tableVO);

        //重新存xml
        tableInfo.setXml(xml);
        tableInfo.setBackXml(backXml);
        tableInfoDao.save(tableInfo);
        sessionUtil.addXml(xml);
        sessionUtil.addXml(backXml);

        //原来的字段信息
        Specification spec = (Specification) (root, criteriaQuery, criteriaBuilder) -> {
            Predicate predicate = criteriaBuilder.equal(root.get("dbId"), dbId);
            return predicate;
        };
        List<ColumnInfo> originalColumnInfos = columnInfoDao.findAll(spec);
        //新的字段信息
        List<ColumnInfo> currentColumnInfos = tableVO.getDataList().stream()
                .map(columnVO -> columnInfo(tableInfo.getId(), tableInfo.getModelName(), columnVO))
                .collect(Collectors.toList());


        //根据ColumnInfo集合得出表实际改动的sql语句集合
        List<String> sqls = getTableSqls(tableInfo.getModelName(), originalColumnInfos, currentColumnInfos);
        //执行sql语句
        for (String sql : sqls) {
            System.out.println(sql);
            jdbcTemplate.execute(sql);

        }
        //备用表同步修改
        List<String> sqlsForBackup = getTableSqls(tableInfo.getBackName(), originalColumnInfos, currentColumnInfos);
        //执行sql语句
        for (String sql : sqlsForBackup) {
            System.out.println(sql);
            jdbcTemplate.execute(sql);

        }

        //重新保存字段信息
        List<ColumnInfo> columnsForAdd = getColumnsForAdd(originalColumnInfos, currentColumnInfos);
        for (ColumnInfo columnInfo : columnsForAdd) {
            columnInfoDao.save(columnInfo);
        }
        List<ColumnInfo> columnsForUpdate = getColumnsForUpdate(originalColumnInfos, currentColumnInfos);
        for (ColumnInfo columnInfo : columnsForUpdate) {
            columnInfoDao.save(columnInfo);
        }
        List<ColumnInfo> columnsFordDelete = getColumnsFordDelete(originalColumnInfos, currentColumnInfos);
        for (ColumnInfo originColumnInfo : columnsFordDelete) {
            columnInfoDao.delete(originColumnInfo);
        }
//            columnInfoDao.deleteAllByDbId(updateTableInfoVO.getDbId());
//
//            List<ColumnInfo> dataList = updateTableInfoVO.getTableVO().getDataList().stream()
//                    .map(columnVO -> columnInfo(tableInfo.getId(), tableInfo.getModelName(), columnVO))
//                    .collect(Collectors.toList());
////            for (ColumnInfo columnInfo : dataList) {
////                columnInfo.setId(null);
////                columnInfo.setPrimaryKey(1);
////
////                columnInfoDao.save(columnInfo);
////            }
//            tableInfo.setUpdatedTime(new Date());
//            String xml2 = createTable(tableVO);
////            重新存xml
//            tableInfo.setXml(xml2);
//            tableInfoDao.save(tableInfo);
        return 1;

    }

    /**
     * 对象类型转换
     * ColumnVo -> ColumnInfo
     */
    private ColumnInfo columnInfo(Integer dbId, String dbName, ColumnVO columnVO) {
        ColumnInfo columnInfo = new ColumnInfo();
        columnInfo.setId(columnVO.getId());
        columnInfo.setFieldName(columnVO.getFieldName());
        columnInfo.setFieldType(columnVO.getFieldType());
        columnInfo.setFieldLength(columnVO.getFieldLength());
        columnInfo.setFieldTitle(columnVO.getFieldTitle());
        columnInfo.setFieldLength(columnVO.getFieldLength());
        columnInfo.setDescription(columnVO.getDescription());
        columnInfo.setDbId(dbId);
        columnInfo.setDbName(dbName);
        columnInfo.setPrimaryKey(columnVO.getPrimaryKey());
        return columnInfo;
    }

    /**
     * 根据ColumnInfo集合得出表实际改动的sql语句集合
     * 根据ColumnInfo的id来匹配字段变化
     */
    private List<String> getTableSqls(String table, List<ColumnInfo> origin, List<ColumnInfo> current) {
        String driverType = env.getProperty("spring.datasource.driver-class-name");
        List<String> result = new ArrayList<>();
        for (ColumnInfo columnInfo : current) {
            ColumnInfo originColumnInfo = origin.stream()
                    .filter(columnInfo1 -> Objects.equals(columnInfo.getId(), columnInfo1.getId()))
                    .findAny()
                    .orElse(null);
            if (isNull(originColumnInfo)) {
                String sql = SqlUtil.addColumn(table, columnInfo.getFieldName(), columnInfo.getFieldType(), columnInfo.getFieldLength());
                result.add(sql);
            } else {
                String sql = SqlUtil.updateColumn(table, originColumnInfo.getFieldName(), columnInfo.getFieldName(), columnInfo.getFieldType(), columnInfo.getFieldLength(), driverType);
                result.add(sql);
            }
        }
        for (ColumnInfo originColumnInfo : origin) {
            if (!originColumnInfo.getFieldName().equals("id")) {
                boolean noneMatch = current.stream()
                        .noneMatch(columnInfo1 -> Objects.equals(originColumnInfo.getId(), columnInfo1.getId()));
                if (noneMatch) {
                    String sql = SqlUtil.deleteColumn(table, originColumnInfo.getFieldName());
                    result.add(sql);
                }
            }

        }
        return result;
    }

    private List<ColumnInfo> getColumnsForAdd(List<ColumnInfo> origin, List<ColumnInfo> current) {
        List<ColumnInfo> result = new ArrayList<>();
        for (ColumnInfo columnInfo : current) {
            ColumnInfo originColumnInfo = origin.stream()
                    .filter(columnInfo1 -> Objects.equals(columnInfo.getId(), columnInfo1.getId()))
                    .findAny()
                    .orElse(null);
            if (isNull(originColumnInfo)) {
                result.add(columnInfo);
            }
        }
        return result;
    }

    private List<ColumnInfo> getColumnsForUpdate(List<ColumnInfo> origin, List<ColumnInfo> current) {
        List<ColumnInfo> result = new ArrayList<>();
        for (ColumnInfo columnInfo : current) {
            ColumnInfo originColumnInfo = origin.stream()
                    .filter(columnInfo1 -> Objects.equals(columnInfo.getId(), columnInfo1.getId()))
                    .findAny()
                    .orElse(null);
            if (nonNull(originColumnInfo)) {
                result.add(columnInfo);
            }
        }
        return result;
    }

    private List<ColumnInfo> getColumnsFordDelete(List<ColumnInfo> origin, List<ColumnInfo> current) {
        List<ColumnInfo> result = new ArrayList<>();
        for (ColumnInfo originColumnInfo : origin) {
            boolean noneMatch = current.stream()
                    .noneMatch(columnInfo1 -> Objects.equals(originColumnInfo.getId(), columnInfo1.getId()));
            if (noneMatch) {
                result.add(originColumnInfo);
            }
        }
        return result;
    }

    @Override
    public int delTable(DelTableVO delTableVO) {
        Optional<TableInfo> byId = tableInfoDao.findById(delTableVO.getId());
        if (!byId.isPresent()) {
            throw new ApiException("此id已经被删除！");
        } else {
            TableInfo tableInfo = byId.get();
            tableInfoDao.deleteById(delTableVO.getId());
            List<ColumnInfo> allByDbId = columnInfoDao.findAllByDbId(delTableVO.getId());
            columnInfoDao.deleteInBatch(allByDbId);
            jdbcTemplate.execute("drop  table  " + tableInfo.getModelName());
            return 1;
        }
    }


    @Override
    public TableAndColumnInfoVO getTableInfoAndColumnInfoByBatch(Integer[] ids) {
        TableAndColumnInfoVO tableAndColumnInfoVO = new TableAndColumnInfoVO();
        PredicateBuilder<TableInfo> builder1 = Specifications.and();
        builder1.in(ids != null, "id", ids);
        List<TableInfo> allTableInfos = tableInfoDao.findAll(builder1.build());

        PredicateBuilder<ColumnInfo> builder2 = Specifications.and();
        builder2.in(ids != null, "dbId", ids);
        List<ColumnInfo> allColumnInfos = columnInfoDao.findAll(builder2.build());

        tableAndColumnInfoVO.setTableInfos(allTableInfos);
        tableAndColumnInfoVO.setColumnInfos(allColumnInfos);
        return tableAndColumnInfoVO;
    }

    public void delTableAndColumnInfo() {
        /* 先删除上次扫描的基础类型*/
        List<TableInfo> tableInfoList = tableInfoDao.findAllByModelType(ModelType.BASICS);
        for (TableInfo tableInfo : tableInfoList) {
            /*删除columnInfo*/
            columnInfoDao.deleteAllByDbId(tableInfo.getId());
        }
    }

    @Override
    public List<Map<String, Object>> executeQuery(String sql) {
        try {
            return jdbcTemplate.queryForList(sql);
        } catch (ApiException e) {
            throw new ApiException("sql语法错误！");
        }

    }

    void changeValueToTargetType(Map map, Object value, Type propertyType, String propertyName) {
        if (propertyType instanceof TimestampType) {
            try {
                if (value instanceof String) {
                    if (StringUtils.isNotEmpty((String) value)) {
                        Date parse = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse((String) value);
                        map.put(propertyName, parse);
                    }
                }
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        if (propertyType instanceof IntegerType) {
            //然后调用强转方法
            int i1 = Integer.valueOf(value + "");
            map.put(propertyName, i1);
        }
        if (propertyType instanceof StringType) {
            //然后调用强转方法
            map.put(propertyName, value + "");
        }
    }


}
