package com.tykj.setting.service;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.tykj.setting.entity.StatisticsSetting;
import com.tykj.setting.entity.vo.StatisticsParam;
import com.tykj.setting.entity.vo.StatisticsSettingVo;
import com.tykj.setting.repository.StatisticsSettingRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class StatisticsSettingService {

    @Autowired
    private StatisticsSettingRepository statisticsSettingRepository;

    public void save(StatisticsSettingVo statisticsSettingVo) {
        boolean newData = Objects.isNull(statisticsSettingVo.getId());
        if (newData) {
            StatisticsSetting statisticsSetting = statisticsSetting(statisticsSettingVo);
            handleAndSave(statisticsSetting);
        } else {
            throw new RuntimeException("新增数据不可附带id");
        }
    }

    public void update(StatisticsSettingVo statisticsSettingVo) {
        boolean exist = Objects.nonNull(statisticsSettingVo.getId()) && statisticsSettingRepository.existsById(statisticsSettingVo.getId());
        if (exist) {
            StatisticsSetting statisticsSetting = statisticsSetting(statisticsSettingVo);
            handleAndSave(statisticsSetting);
        } else {
            throw new RuntimeException("未找到该id的数据");
        }
    }

    public List<StatisticsSettingVo> findByLimit(Integer limit) {
        List<StatisticsSetting> result = statisticsSettingRepository.findAllByOrderPosLessThanEqual(limit);
        return result.stream()
                .map(this::statisticsSettingVo)
                .sorted(Comparator.comparingInt(StatisticsSettingVo::getOrder))
                .collect(Collectors.toList());
    }

    public List<StatisticsSettingVo> findAll() {
        return statisticsSettingRepository.findAll().stream()
                .map(this::statisticsSettingVo)
                .collect(Collectors.toList());
    }

    public StatisticsSettingVo findById(Integer id) {
        return statisticsSettingRepository.findById(id)
                .map(this::statisticsSettingVo)
                .orElseThrow(() -> new RuntimeException("未找到该id的数据"));
    }

    public void deleteById(Integer id) {
        statisticsSettingRepository.deleteById(id);
    }

    private void handleAndSave(StatisticsSetting statisticsSetting) {
        Integer order = statisticsSetting.getOrderPos();
        StatisticsSetting origin = statisticsSettingRepository.findByOrderPos(order);
        if (Objects.nonNull(origin)) {
            origin.setOrderPos(null);
            statisticsSettingRepository.save(origin);
        }
        statisticsSettingRepository.save(statisticsSetting);
    }


    private StatisticsSetting statisticsSetting(StatisticsSettingVo statisticsSettingVo) {
        String params = "[]";
        try {
            params = new ObjectMapper().writeValueAsString(statisticsSettingVo.getParams());
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        return new StatisticsSetting(
                statisticsSettingVo.getId(),
                statisticsSettingVo.getName(),
                statisticsSettingVo.getType(),
                statisticsSettingVo.getOrder(),
                params
        );
    }

    private StatisticsSettingVo statisticsSettingVo(StatisticsSetting statisticsSetting) {
        List<StatisticsParam> params = new ArrayList<>();
        try {
            params = new ObjectMapper().readValue(statisticsSetting.getParams(), new TypeReference<List<StatisticsParam>>() {
            });
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        return new StatisticsSettingVo(
                statisticsSetting.getId(),
                statisticsSetting.getName(),
                statisticsSetting.getType(),
                statisticsSetting.getOrderPos(),
                params
        );
    }
}
