/*
 * Copyright 2004, 2005, 2006 Acegi Technology Pty Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tykj.user.authencation.filter;


import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.Nullable;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 用户名密码验证拦截器
 *
 * @author HuangXiahao
 * @version V1.0
 * @class CustomJWTAuthenticationFilter
 * @packageName com.example.personnelmanager.common.authencation.filter
 * @data 2020/6/13
 **/
public class CustomUsernamePasswordAuthenticationFilter extends
        AbstractAuthenticationProcessingFilter {
    @Autowired
    SuccessHandler healder;

    public static final String SPRING_SECURITY_FORM_USERNAME_KEY = "username";
    public static final String SPRING_SECURITY_FORM_PASSWORD_KEY = "password";

    public CustomUsernamePasswordAuthenticationFilter() {
        //设置用户接口的路径以及访问方式
        super(new AntPathRequestMatcher("/user/login", "POST"));
    }

    /***
     * 在这个方法中执行验证操作
     * @param request
     * @param response
     * @Return : org.springframework.security.core.Authentication
     */
    @Override
    public Authentication attemptAuthentication(HttpServletRequest request,
                                                HttpServletResponse response) throws AuthenticationException {
        String username = obtainUsername(request);
        String password = obtainPassword(request);
        if (username == null) {
            username = "";
        }
        if (password == null) {
            password = "";
        }
        username = username.trim();
        //生成springSecurity能够辨认的用户类型
        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(
                username, password);
        //为用于验证的用户注入session信息
        setDetails(request, authRequest);
        //进行验证
        return getAuthenticationManager().authenticate(authRequest);
    }

    @Override
    protected AuthenticationSuccessHandler getSuccessHandler() {
        return healder;
    }

    /**
     * 从request中取出密码
     * @param request
     * @return
     */
    @Nullable
    protected String obtainPassword(HttpServletRequest request) {
        return request.getParameter(SPRING_SECURITY_FORM_PASSWORD_KEY);
    }

    /**
     * 从request中取出用户名
     *
     * @param request
     * @return
     */
    @Nullable
    protected String obtainUsername(HttpServletRequest request) {
        return request.getParameter(SPRING_SECURITY_FORM_USERNAME_KEY);
    }

    /***
     * 为用户注入session信息
     * @param request
     * @param authRequest
     * @Return : void
     */
    protected void setDetails(HttpServletRequest request,
                              UsernamePasswordAuthenticationToken authRequest) {
        authRequest.setDetails(authenticationDetailsSource.buildDetails(request));
    }

}
