package com.tykj.user.service.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.tykj.base.result.ResultObj;
import com.tykj.user.dao.OrganizationDao;
import com.tykj.user.dao.StorageKeyDao;
import com.tykj.user.dao.UserDao;
import com.tykj.user.pojo.Organization;
import com.tykj.user.pojo.StorageKey;
import com.tykj.user.pojo.User;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import static java.util.Objects.nonNull;

@Slf4j
@Service
public class SyncDataService {

    @Autowired
    private OrganizationDao organizationDao;
    @Autowired
    private UserDao userDao;
    @Autowired
    private StorageKeyDao storageKeyDao;

    private RestTemplate restTemplate = new RestTemplate();

    @Value("${sync.ip}")
    private String ip;

    @Value("${sync.port}")
    private Integer port;


    @Scheduled(cron = "0 0 */1 * * ?")
    public void execute() {
        syncKey();
        syncUser();
        syncOrganization();
    }

    private void syncKey() {
        log.info("开始同步秘钥信息...");
        String url = String.format("http://%s:%s", ip, port);
        ResponseEntity<String> entity = restTemplate.exchange(url + "/storageKey/getKeys", HttpMethod.POST, getEntity(), String.class, new HashMap<>());
        if (nonNull(entity) && nonNull(entity.getBody())) {
            try {
                ResultObj<StorageKey> resultObj = new ObjectMapper().readValue(entity.getBody(), new TypeReference<ResultObj<StorageKey>>() {
                });
                StorageKey key = resultObj.getData();
                storageKeyDao.save(key);
            } catch (JsonProcessingException e) {
                e.printStackTrace();
                log.error("解析json数据失败");
            }
        }
    }

    private void syncUser() {
        log.info("开始同步用户信息...");
        String url = String.format("http://%s:%s", ip, port);
        ResponseEntity<String> entity = restTemplate.exchange(url + "/user/all", HttpMethod.GET, getEntity(), String.class, new HashMap<>());
        if (nonNull(entity) && nonNull(entity.getBody())) {
            try {
                ResultObj<List<User>> resultObj = new ObjectMapper().readValue(entity.getBody(), new TypeReference<ResultObj<List<User>>>() {
                });
                List<User> users = resultObj.getData();
                userDao.saveAll(users);
            } catch (JsonProcessingException e) {
                e.printStackTrace();
                log.error("解析json数据失败");
            }
        }
    }

    private void syncOrganization() {
        log.info("开始同步组织架构信息...");
        String url = String.format("http://%s:%s", ip, port);
        ResponseEntity<String> entity = restTemplate.exchange(url + "/organization/all", HttpMethod.GET, getEntity(), String.class, new HashMap<>());
        if (nonNull(entity) && nonNull(entity.getBody())) {
            try {
                ResultObj<List<Organization>> resultObj = new ObjectMapper().readValue(entity.getBody(), new TypeReference<ResultObj<List<Organization>>>() {
                });
                List<Organization> organizations = resultObj.getData();
                organizationDao.saveAll(organizations);
            } catch (JsonProcessingException e) {
                e.printStackTrace();
                log.error("解析json数据失败");
            }
        }
    }

    private HttpEntity getEntity() {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Collections.singletonList(MediaType.APPLICATION_JSON));
        return new HttpEntity<>(null, headers);
    }
}
