package com.example.personnelmanager.common.config;

import com.example.personnelmanager.entity.UserDetail;
import com.example.personnelmanager.service.UserServer;
import com.google.gson.Gson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.stereotype.Service;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;


/**
 * 描述：sprinSecurity安全框架配置类
 * @author HuangXiahao
 * @version V1.0
 * @class Java8TimeConfig
 * @packageName com.example.personnelmanager.common.config
 * @data 2020/5/20
 **/

@Configuration
@EnableWebSecurity
@EnableGlobalMethodSecurity(jsr250Enabled = true)
public class SecurityWebConfig extends WebSecurityConfigurerAdapter {

    @Bean
    public PasswordEncoder passwordEncoder() {
        return new BCryptPasswordEncoder();
    }

    @Override
    protected void configure(HttpSecurity http) throws Exception {
        http.cors().and().formLogin()
                .loginProcessingUrl("/api/user/login")
                .successHandler(new CustomAuthenticationSuccessHandler())
                .failureHandler(new CustomAuthenticationFailureHandler())
                .permitAll()
                .usernameParameter("username")
                .passwordParameter("password")
                .and()
                .authorizeRequests()
//                .antMatchers("/user/*").hasRole("NORMAL")
                .anyRequest()
//                .authenticated()
                .permitAll()
                .and()
                .logout()
                .logoutUrl("/user/logout").permitAll()
                .and()
                .sessionManagement()
                .and()
                .csrf().disable();
        http.exceptionHandling()
                .authenticationEntryPoint(new CustomAuthenticationEntryPoint());
    }

    @Override
    protected void configure(AuthenticationManagerBuilder auth) throws Exception {
        auth.userDetailsService(new MyUserDetailsService()).passwordEncoder(passwordEncoder());
        super.configure(auth);
    }

    @Service
    public class MyUserDetailsService implements UserDetailsService {

        @Autowired
        UserServer userServer;

        @Override
        public UserDetails loadUserByUsername(String username) throws UsernameNotFoundException {
            return   userServer.selectByUserName(username);
        }
    }

    /***
     * 登录成功后干些啥
     */
    public static class CustomAuthenticationSuccessHandler implements AuthenticationSuccessHandler {

        @Override
        public void onAuthenticationSuccess(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse,
                                            Authentication authentication) throws IOException{
            httpServletResponse.setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
            ServletOutputStream servletOutputStream = httpServletResponse.getOutputStream();
            UserDetail user = (UserDetail) authentication.getPrincipal();
            Map<String, Object> result = new HashMap<>(3);
            result.put("username", user.getUsername());
            result.put("mgs","登录成功");
            servletOutputStream.write(new Gson().toJson(result).getBytes("utf-8"));
            servletOutputStream.flush();
            servletOutputStream.close();
        }
    }

    /**
     * 登录失败了干些啥
     */
    public static class CustomAuthenticationFailureHandler implements AuthenticationFailureHandler {

        @Override
        public void onAuthenticationFailure(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse,
                                            AuthenticationException e) throws IOException, ServletException {
            httpServletResponse.setHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_UTF8_VALUE);
            httpServletResponse.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
            ServletOutputStream servletOutputStream = httpServletResponse.getOutputStream();
            Map<String, Object> result = new HashMap<>(1);
            result.put("msg", e.getMessage());
            servletOutputStream.write(new Gson().toJson(result).getBytes("utf-8"));
            servletOutputStream.flush();
            servletOutputStream.close();
        }
    }

    /**
     * 没有登录干些啥
     */
    public static class CustomAuthenticationEntryPoint implements AuthenticationEntryPoint {

        @Override
        public void commence(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse,
                             AuthenticationException e) throws IOException, ServletException {
            httpServletResponse.setStatus(HttpStatus.UNAUTHORIZED.value());
        }
    }

}
