package com.example.personnelmanager.controller;

import com.example.personnelmanager.common.exception.GlobalException;
import com.example.personnelmanager.common.validIGroup.GroupSave;
import com.example.personnelmanager.common.validIGroup.GroupUpdate;
import com.example.personnelmanager.entity.Certificate;
import com.example.personnelmanager.entity.CertificateImage;
import com.example.personnelmanager.entity.commonEntity.ResultObj;
import com.example.personnelmanager.entity.vo.pageableVo.CertificatePageableVo;
import com.example.personnelmanager.service.CertificateImageService;
import com.example.personnelmanager.service.CertificateService;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.constraints.NotNull;
import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.UUID;

/**
 * 人员证书控制器
 *
 * @author HuangXiahao
 * @version V1.0
 * @class CertificateController
 * @packageName com.example.personnelmanager.controller
 * @data 2020/5/18
 **/
@RestController
@RequestMapping("/api/certificate")
public class CertificateController {

    /**
     * 图片保存路径
     */
    @Value("${file.imagePath}")
    String imagesFilePath;

    final CertificateService certificateService;

    final CertificateImageService certificateImageService;

    public CertificateController(CertificateService certificateService, CertificateImageService certificateImageService) {
        this.certificateService = certificateService;
        this.certificateImageService = certificateImageService;
    }

    /**
     * 新增证书
     * @param certificate
     * @return ResultObj
     */
    @ApiOperation(value = "新增证书", notes = "为某一个人员新增其名下的证书", httpMethod = "POST")
    @RequestMapping(method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResultObj addCertificate(@Validated(GroupSave.class) @RequestBody Certificate certificate) {
        Certificate addCertificate = certificateService.addEntity(certificate);
        if (addCertificate != null){
            return new ResultObj(addCertificate,"插入成功");
        }else {
            throw new GlobalException("插入失败，出现未知的失败原因");
        }
    }

    /**
     * 带分页的人员证书列表
     * @param certificateVo
     * @return ResultObj
     */
    @RequestMapping(value = "/list",method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResultObj getCertificatePage(@RequestBody CertificatePageableVo certificateVo) {
        return new ResultObj(certificateService.getPage(certificateVo.getCertificate(), certificateVo.getPageable()), "success");
    }

    /**
     * 删除人员证书
     * @param certificateId
     * @return ResultObj
     */
    @RequestMapping(method = RequestMethod.DELETE)
    public ResultObj delCertificate(@NotNull Long certificateId) {
        return new ResultObj(certificateService.delete(certificateId), "success");
    }

    /**
     * 获取单个人员证书信息
     * @param certificateId
     * @return ResultObj
     */
    @RequestMapping(method = RequestMethod.GET)
    public ResultObj getContract(@NotNull Long certificateId) {
        return new ResultObj(certificateService.getOne(certificateId), "success");
    }

    /**
     * 更新人员证书信息
     * @param certificate
     * @return ResultObj
     */
    @RequestMapping(method = RequestMethod.PUT, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResultObj updateContract(@Validated(GroupUpdate.class) @RequestBody Certificate certificate) {
        return new ResultObj(certificateService.update(certificate), "success");
    }

    /**
     * 删除人员证书信息
     * @param certificateImageId
     * @return ResultObj
     */
    @RequestMapping(value = "/deleteImage",method = RequestMethod.DELETE)
    public ResultObj delContractImage(@NotNull Long certificateImageId) {
        return new ResultObj(certificateImageService.delete(certificateImageId), "success");
    }

    /***
     * 上传人员证书图片
     * @param file
     * @param certificateId
     * @Return : com.example.personnelmanager.entity.commnEntity.ResultObj
    */
    @RequestMapping(value = "/file",method = RequestMethod.POST)
    public ResultObj addContractImage(@RequestParam("file") MultipartFile file, @NotNull Long certificateId) throws IOException {
        if (!file.isEmpty()) {
            //新建人员证书图片类
            CertificateImage certificateImage = new CertificateImage();
            //获取文件名
            String fileName = file.getOriginalFilename();
            //获取文件后缀
            String suffixName = Objects.requireNonNull(fileName).substring(fileName.lastIndexOf("."));
            //重新生成文件名
            fileName = UUID.randomUUID() + suffixName;
            //指定本地存入路径
            file.transferTo(new File(imagesFilePath + fileName));
            //为图片设置对应的人员证书
            Certificate certificate = new Certificate();
            certificate.setCertificateId(certificateId);
            certificateImage.setCertificate(certificate);
            //为图片设置本地存储路径
            certificateImage.setUrl("/images/" + fileName);
            //存入数据库
            certificateImageService.addEntity(certificateImage);
            return new ResultObj(certificateImage, "success");
        } else {
            throw new GlobalException("文件是空的");
        }
    }

}
