package com.example.personnelmanager.service.impl;

import com.example.personnelmanager.common.exception.GlobalException;
import com.example.personnelmanager.common.utils.AuthenticationUtils;
import com.example.personnelmanager.dao.OrganizationNodeRepository;
import com.example.personnelmanager.dao.PostRepository;
import com.example.personnelmanager.entity.OrganizationNode;
import com.example.personnelmanager.entity.vo.OrganizationNodeListVo;
import com.example.personnelmanager.entity.UserDetail;
import com.example.personnelmanager.entity.vo.PostVo;
import com.example.personnelmanager.service.OrganizationNodeService;
import com.github.wenhao.jpa.PredicateBuilder;
import com.github.wenhao.jpa.Specifications;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author HuangXiahao
 * @data 2020/5/15
 **/

@Service
@Transactional
public class OrganizationNodeServiceImpl implements OrganizationNodeService {

    @Autowired
    OrganizationNodeRepository organizationNodeRepository;

    @Autowired
    PostRepository postRepository;

    @Override
    public OrganizationNode addEntity(OrganizationNode organizationNode) {
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        if (organizationNode.getParentNode()!=null){
            OrganizationNode one = getOne(organizationNode.getParentNode());
            if (!one.getEnterprise().getEnterpriseId().equals(userDetails.getEnterprise().getEnterpriseId())){
                throw new UsernameNotFoundException("当前登录用户无权限操作该父节点");
            }
            organizationNode.setNodeLevel(one.getNodeLevel()+one.getNodeId()+".");
        }else {
            organizationNode.setNodeLevel("0.");
        }
        organizationNode.setEnterprise(userDetails.getEnterprise());
        OrganizationNode save = organizationNodeRepository.save(organizationNode);
        return save;
    }

    @Override
    public Page getPage(OrganizationNode organizationNode, Pageable pageable) {
        Specification<OrganizationNode> specification = specificationBuild(organizationNode);
        return (Page<OrganizationNode>) organizationNodeRepository.findAll(specification, pageable);
    }

    @Override
    public List<OrganizationNode> getList(OrganizationNode organizationNode) {
        Specification<OrganizationNode> specification = specificationBuild(organizationNode);
        return organizationNodeRepository.findAll(specification);
    }

    @Override
    public List<OrganizationNodeListVo> getTree(){
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        List<OrganizationNodeListVo> organizationNodeListVos = organizationNodeRepository.findTreeNode(userDetails.getEnterprise().getEnterpriseId());
        Map<Long, OrganizationNodeListVo> nodeCollect =
               organizationNodeListVos.stream().collect(Collectors.toMap(OrganizationNodeListVo::getNodeId,organizationNodeListVo -> organizationNodeListVo));
        List<OrganizationNodeListVo> organizationNodeListVos1 = parseTreeFromDown(
                organizationNodeListVos,
                OrganizationNodeListVo::getNodeId,
                organizationNodeListVo -> Optional.ofNullable(nodeCollect.get(organizationNodeListVo.getParentNode())),
                OrganizationNodeListVo::addChildNode
        );
        return organizationNodeListVos1;
    }

    /**
     * 自底向上整理出树结构，并存放在Node对象中
     * 利用Map将原始List的对象的位置重新整理，并将所有的指针都设置好
     *
     * @param originList      原始待整理的列表
     * @param getId           列表里每个对象获取自身id的方法，用于避免重复计算
     * @param getParent       列表里每个对象获取父对象的方法
     * @param targetSetChild  用于最终对象设置子节点的方法，形如R.setChild(R),或者R.addChildList(R)
     * @param <V>             原始对象
     * @param <T>             主键类型
     * @return 根节点对象集合
     */
    public <V, T extends Number> List<V> parseTreeFromDown(List<V> originList,
                                                              Function<V, T> getId,
                                                              Function<V, Optional<V>> getParent,
                                                              BiConsumer<V, V> targetSetChild) {
        //K为主键id , Value 为最终对象
        Map<T, V> map = new HashMap<>(32);
        List<T> rootIds = new ArrayList<>();

        for (V originNode : originList) {
            //对于所有节点，如果已经遍历过了则直接取已经设置过子节点的引用
            V targetNode = map.getOrDefault(getId.apply(originNode), originNode);

            Optional<V> parentNode = getParent.apply(originNode);
            //查询父节点，如果不存在父节点则证明该节点为根节点，直接放入map中
            if (parentNode.isPresent()) {
                //否则查询该父节点是否已经已经被连接过指针，如果连接过则取出连接过的继续连接，否则进行第一次连接并存入map
                V parent = parentNode.get();
                T parentId = getId.apply(parent);
                V parentInMap = map.get(parentId);
                if (parentInMap == null) {
                    targetSetChild.accept(parent, targetNode);
                    map.put(parentId, parent);
                } else {
                    targetSetChild.accept(parentInMap, targetNode);
                }
                //连接完处理之后还需要将自身这个节点存入map
                map.put(getId.apply(originNode), targetNode);
            } else {
                //root node
                map.put(getId.apply(originNode), targetNode);
                rootIds.add(getId.apply(originNode));
            }

        }

        //根据rootIds返回所有的顶层节点
        return rootIds.stream().map(map::get).collect(Collectors.toList());
    }


    @Override
    public OrganizationNode update(OrganizationNode organizationNode) {
        Optional<OrganizationNode> byId = organizationNodeRepository.findById(organizationNode.getNodeId());
        if (!byId.isPresent()) {
            throw new GlobalException("所要修改的节点不存在");
        }
        OrganizationNode one = byId.get();
        if (one.getParentNode()== 0L){
            if (organizationNode.getParentNode()!= 0L){
                throw new GlobalException("根节点不能被移动");
            }
        }
        OrganizationNode after = organizationNodeRepository.save(organizationNode);
        return after;
    }

    @Override
    public OrganizationNode getOne(Long id) {
        Optional<OrganizationNode> byId = organizationNodeRepository.findById(id);
        if (!byId.isPresent()) {
            throw new GlobalException("所要查询的节点不存在");
        }
        return byId.get();
    }

    @Override
    public Boolean delete(Long id) {
        Optional<OrganizationNode> byId = organizationNodeRepository.findById(id);
        if (!byId.isPresent()) {
            throw new GlobalException("不存在需要被删除的节点");
        }
        OrganizationNode one = byId.get();
        if (one.getParentNode()== 0L){
            throw new GlobalException("根结点不能被删除");
        }
        //删除节点以及所有子节点
        organizationNodeRepository.deleteorganizationNode(id);
        //删除所有被删除节点下的岗位
        postRepository.deletePostByNodeId(id);
        return true;
    }

    @Override
    public Specification<OrganizationNode> specificationBuild(OrganizationNode organizationNode) {
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        PredicateBuilder<OrganizationNode> and = Specifications.and();
        and.eq("enterprise", userDetails.getEnterprise());
        and.eq("deleteTag", 0);
        return and.build();
    }
}
