package com.example.personnelmanager.service.impl;

import com.example.personnelmanager.common.exception.GlobalException;
import com.example.personnelmanager.common.utils.AuthenticationUtils;
import com.example.personnelmanager.common.utils.DateCalculationUtil;
import com.example.personnelmanager.dao.PeopleRepository;
import com.example.personnelmanager.entity.*;
import com.example.personnelmanager.entity.vo.PeopleVo;
import com.example.personnelmanager.entity.vo.pageableVo.PeoplePageableVo;
import com.example.personnelmanager.service.AgainEntryInformationService;
import com.example.personnelmanager.service.BecomeRegularWorkerService;
import com.example.personnelmanager.service.LeaveInformationService;
import com.example.personnelmanager.service.PeopleService;
import com.github.wenhao.jpa.PredicateBuilder;
import com.github.wenhao.jpa.Specifications;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.security.core.parameters.P;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.Instant;
import java.util.List;
import java.util.Optional;

/**
 * @author HuangXiahao
 * @version V1.0
 * @class LegalEntityServiceImpl
 * @packageName com.example.personnelmanager.service
 * @data 2020/5/13
 **/

@Service
public class PeopleServiceImpl implements PeopleService {

    @Autowired
    PeopleRepository peopleRepository;

    @Autowired
    LeaveInformationService leaveInformationService;

    @Autowired
    AgainEntryInformationService againEntryInformationService;

    @Autowired
    BecomeRegularWorkerService becomeRegularWorkerService;

    @Override
    public People addEntity(People people) {
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        if (peopleRepository.findByPhoneAndEnterprise(people.getPhone(),userDetails.getEnterprise().getEnterpriseId()).size()>0){
            throw new GlobalException("手机号码与其他人员重复");
        }
        if (people.getTrialEndTime()!=null&&people.getInternshipEndTime().compareTo(Instant.now())>0){
            people.setJobStatus("试用");
        }else if (people.getInternshipEndTime()!=null&&people.getInternshipEndTime().compareTo(Instant.now())>0){
            people.setJobStatus("实习");

        }else {
            people.setJobStatus("在职");
        }
        if (people.getBirthday()!=null){
            people.setAge((int) DateCalculationUtil.twoTimeYearCalculation(people.getBirthday(),Instant.now()));
        }
        people.setEnterprise(userDetails.getEnterprise());
        People save = peopleRepository.save(people);
        return save;
    }

    @Override
    public Page getPage(People people, Pageable pageable) {
        Specification<People> specification = specificationBuild(people);
        return  peopleRepository.findAll(specification, pageable);
    }

    @Override
    public List<People> getList(People people) {
        Specification<People> specification = specificationBuild(people);
        return peopleRepository.findAll(specification);
    }

    @Override
    public People update(People people) {
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        people.setEnterprise(userDetails.getEnterprise());
        if (people.getEnterprise()!=null&& !userDetails.getEnterprise().getEnterpriseId().equals(people.getEnterprise().getEnterpriseId())){
            throw new GlobalException("用户无权限对该人员进行操作");
        }
        if (peopleRepository.findByPeopleIdAndPhoneAndEnterprise(people.getPeopleId(),people.getPhone(),userDetails.getEnterprise().getEnterpriseId()).size()>0){
            throw new GlobalException("手机号码与其他人员重复");
        }
        if (people.getBirthday()!=null){
            people.setAge((int) DateCalculationUtil.twoTimeYearCalculation(people.getBirthday(),Instant.now()));
        }
        People after = peopleRepository.save(people);
        return after;
    }

    @Override
    public People getOne(Long id) {
        Optional<People> byId = peopleRepository.findById(id);
        if (!byId.isPresent()) {
            throw new GlobalException("所要查询的人员不存在");
        }
        return byId.get();
    }

    @Override
    public Boolean delete(Long id) {
        Optional<People> byId = peopleRepository.findById(id);
        if (!byId.isPresent()) {
            throw new GlobalException("不存在需要被删除的人员");
        }
        People people = byId.get();
        people.setDeleteTag(1);
        peopleRepository.save(people);
        return true;
    }

    @Override
    public Specification<People> specificationBuild(People people) {
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        PredicateBuilder and = Specifications.<People>and();
        and.eq("enterprise.enterpriseId", userDetails.getEnterprise().getEnterpriseId());
        and.eq("deleteTag", 0);
        return and.build();
    }


    @Override
    @Transactional
    public People leave(LeaveInformation leaveInformation) {
        People one = getOne(leaveInformation.getPeople().getPeopleId());
        one.setJobStatus("离职");
        People update = update(one);
        leaveInformationService.addEntity(leaveInformation);
        return update;
    }

    @Override
    @Transactional
    public People againEntry(AgainEntryInformation againEntryInformation) {
        People one = getOne(againEntryInformation.getPeople().getPeopleId());
        if (!one.getJobStatus().equals("离职")){
            throw new GlobalException("人员非离职状态无法再次入职");
        }
        one.setJobStatus("在职");
        People update = update(one);
        againEntryInformationService.addEntity(againEntryInformation);
        return update;
    }

    /**
     * 描述：人员转正，并添加转正记录
     * @param becomeRegularWorkerInformation
     * @return
     */
    @Override
    @Transactional
    public People  becomeRegularWorker(BecomeRegularWorkerInformation becomeRegularWorkerInformation) {
        People one = getOne(becomeRegularWorkerInformation.getPeople().getPeopleId());
        if (one.getJobStatus().equals("实习")||one.getJobStatus().equals("试用")){
            one.setJobStatus("在职");
            People update = update(one);
            becomeRegularWorkerService.addEntity(becomeRegularWorkerInformation);
            return update;
        }else {
            throw new GlobalException("人员非实习或者试用不能转正");
        }
    }

    @Override
    public Page getList(PeoplePageableVo peoplePageableVo) {
        Specification specification = specificationBuilder(peoplePageableVo);
        Page<People> all = peopleRepository.findAll(specification, peoplePageableVo.getPageable());
        return all;
    }

    public Specification<People> specificationBuilder(PeoplePageableVo peopleDto){
        UserDetail userDetails = AuthenticationUtils.getAuthentication();
        PredicateBuilder<People> and = Specifications.and();
        if (!StringUtils.isEmpty(peopleDto.getName())){
            and.like("name","%"+peopleDto.getName()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getSex())){
            and.like("sex","%"+peopleDto.getSex()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getIdCardNumber())){
            and.like("idCardNumber","%"+peopleDto.getIdCardNumber()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getJobStatus())){
            and.like("jobStatus","%"+peopleDto.getJobStatus()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getJobStatus())){
            and.like("jobStatus","%"+peopleDto.getJobStatus()+"%");
        }
        if (peopleDto.getStartAge()!=null){
            and.ge("age",peopleDto.getStartAge());
        }
        if (peopleDto.getEndAge()!=null){
            and.le("age",peopleDto.getEndAge());
        }
        if (!StringUtils.isEmpty(peopleDto.getPhone())){
            and.like("phone","%"+peopleDto.getPhone()+"%");
        }
        if (peopleDto.getStartWorkAge()!=null){
            and.ge("workingAge",peopleDto.getStartAge());
        }
        if (peopleDto.getEndWorkAge()!=null){
            and.le("workingAge",peopleDto.getEndAge());
        }
        if (!StringUtils.isEmpty(peopleDto.getPoliticCountenance())){
            and.like("politicCountenance","%"+peopleDto.getPoliticCountenance()+"%");
        }
        if (peopleDto.getStartBirthday()!=null){
            and.le("birthday",peopleDto.getStartBirthday());
        }
        if (peopleDto.getEndBirthday()!=null){
            and.le("birthday",peopleDto.getEndBirthday());
        }
        if (!StringUtils.isEmpty(peopleDto.getHighestEducation())){
            and.like("highestEducation","%"+peopleDto.getHighestEducation()+"%");
        }
        if (peopleDto.getStartEntryTime()!=null){
            and.le("entryTime",peopleDto.getStartEntryTime());
        }
        if (peopleDto.getEndEntryTime()!=null){
            and.le("entryTime",peopleDto.getEndEntryTime());
        }
        if (!StringUtils.isEmpty(peopleDto.getTitleLevel())){
            and.like("titles.titleLevel","%"+peopleDto.getTitleLevel()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getTitleName())){
            and.like("titles.titleName","%"+peopleDto.getTitleName()+"%");
        }
        if (!StringUtils.isEmpty(peopleDto.getCertificateName())){
            and.like("certificates.certificateName","%"+peopleDto.getCertificateName()+"%");

        }
        and.eq("enterprise.enterpriseId", userDetails.getEnterprise().getEnterpriseId());
        and.eq("deleteTag", 0);
        return and.build();
    }
}
