package snowflake;

import java.util.concurrent.atomic.AtomicLong;

/**
 * 描述：分布式自增ID算法
 * ID共64位
 * @author HuangXiahao
 * @version V1.0
 * @class SnowFake
 * @packageName my.test.hello.snowflake
 * @data 2020/5/20
 **/

public class SnowFlake {

    /**
     * 机器ID位数
     * */
    private long workIdLength ;

    /**
     * 每秒产生的序列号位数
     * */
    private long sequenceLength ;

    /**
     * 每秒产生的序列号最大值
     **/
    private long maxSequence = ~(-1L <<sequenceLength);

    /**
     * 默认给了一个机器码
     * */
    private final long workId = 116L;

    /**
     * 用于记录上一毫秒的时间戳
     * */
    private long lastTimestamp = 0L;

    /**
     * 用于记录上一个序列号
     * */
    private long sequence = 0L;

    /**
     * 时间戳左编译量
     * */
    private final long timestampOffset = sequenceLength + workIdLength;

    /**
     * 计数器
     * */
    private AtomicLong atomicLong = new AtomicLong(0);

    public SnowFlake(long sequenceLength) {
        this.sequenceLength = sequenceLength;
        this.workIdLength = 22L - sequenceLength;
        maxSequence = ~(-1L <<sequenceLength);
    }

    /***
     * 描述：获取雪花算法ID
     * @Retrun : long
     * @Author : HuangXiahao
     * @Date : 2020/5/22 15:31
    */
    public synchronized long nextId(){
        //获取当前时间戳
        long currentTime = getNewTime();
        if (lastTimestamp==currentTime){
            sequence = (sequence + 1) & maxSequence;
            if (sequence==0L){
                //进入下一毫秒并将序列号重置
                currentTime = nextTime(lastTimestamp);
            }
        }else {
            //重置时间戳记录和序列号
            sequence = 0L;
        }
        lastTimestamp = currentTime;
        atomicLong.incrementAndGet();
        //将数据移动到对应的位置上
        return currentTime<<(timestampOffset)|workId<<(sequenceLength)|sequence;
    }

    /***
     * 描述：进入下一毫秒并重置序列号和时间记录
     * @Return : void
     * @Author : HuangXiahao
     * @Date : 2020/5/22 15:39
    */
    public Long nextTime(long previouTimestamp){
        long currentTime = getNewTime();
        while (currentTime<=previouTimestamp){
            currentTime = getNewTime();
        }
        return currentTime;
    }

    private Long getNewTime(){
        return System.currentTimeMillis();
    }

    /**
     * 测试生成N个雪花ID中是否出现错误ID
     * @param n 生成个数
     */
    public void testError(int n){
        long last = 0L;
        for (int i = 0; i <n ; i++) {
            long l = nextId();
            if (last!=l){
                if (last>l&&last!=0){
                    System.out.println("非单调递增");
                }
                last = l;
            }else {
                System.out.println("重复了");
            }
        }
        System.out.println("跑完了");
    }

    /**
     * 测试生成N个雪花ID的QPS
     * @param n 生成个数
     */
    public void testQPS(int n){
        long start = System.currentTimeMillis();
        for (int i = 0; i < n ; i++) {
            nextId();
        }
        long end = System.currentTimeMillis();
        System.out.println("每秒QPS："+(n/(end-start))*1000 );
        System.out.println(end-start+"ms");
    }

    public long getAtomicLong() {
        return atomicLong.get();
    }
}
