package com.tykj.workflowcore.base.result;

import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;


/**
 * 错误处理类
 * 所有的报错信息都会通过本层的方法向外界返回
 *
 * @author HuangXiahao
 **/
@ControllerAdvice
@Slf4j
public class GlobalExceptionHandler {

    /**
     * 报错信息都会通过这个方法处理并通过统一的返回方式进行返回
     *
     * @param e 报错信息
     */
    @ResponseBody
    @ExceptionHandler(Exception.class)
    public ResponseEntity errorMessage(Exception e) {
        log.error("[其他异常] {}", e.toString());
        e.printStackTrace();
        return ResultUtil.failed(e.getMessage());
    }

    /**
     * 参数校验错误
     * @param e 报错信息
     */
    @ResponseBody
    @ExceptionHandler(MethodArgumentNotValidException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity errorMessage(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        String errorMesssage = "参数校验失败:";
        for (FieldError fieldError : bindingResult.getFieldErrors()) {
            errorMesssage += fieldError.getDefaultMessage() + ", ";
        }
        log.error("[参数校验异常] {}", errorMesssage);
        e.printStackTrace();
        return ResultUtil.failed(errorMesssage);
    }

    /**
     * 业务错误
     *
     * @param e 报错信息
     */
    @ResponseBody
    @ExceptionHandler(ApiException.class)
    public ResponseEntity errorMessage(ApiException e) {
        log.warn("[自定义异常] {}", e.toString());
        if (e.getResponseEntity() != null) {
            return e.getResponseEntity();
        }
        return ResultUtil.failed(e.getMessage());
    }

    /**
     * 数据合法性验证报错会通过这个方法处理并通过统一的返回方式进行返回
     *
     * @param e 报错信息
     */
    @ResponseBody
    @ExceptionHandler(BindException.class)
    public ResponseEntity errorMessage(BindException e) {
        log.error("[参数异常] 检测到用户访问接口没有提供正确的参数 {}", e.toString());
        e.printStackTrace();
        BindingResult bindingResult = e.getBindingResult();
        String message = null;
        if (bindingResult.hasErrors()) {
            FieldError fieldError = bindingResult.getFieldError();
            if (fieldError != null) {
                message = fieldError.getField() + fieldError.getDefaultMessage();
            }
        }
        return ResultUtil.failed(message);
    }


}

