package com.tykj.workflowcore.model_layer.utils;


import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * @author C
 */
@Slf4j
public class SqlUtil {

    /**
     * java类型对应sql类型Map
     */
    private static final Map<String, String> TYPE_MAP = new HashMap<>();

    static {
        TYPE_MAP.put("java.lang.Integer", "int");
        TYPE_MAP.put("java.lang.Long", "bigint");
        TYPE_MAP.put("java.lang.Short", "smallint");
        TYPE_MAP.put("java.lang.Float", "float");
        TYPE_MAP.put("java.lang.Double", "double");
        TYPE_MAP.put("java.lang.BigDecimal", "numeric");
        TYPE_MAP.put("java.lang.Character", "char(1)");
        TYPE_MAP.put("java.lang.String", "varchar");
        TYPE_MAP.put("java.lang.Byte", "tinyint");
        TYPE_MAP.put("java.lang.Boolean", "bit");
        TYPE_MAP.put("java.lang.Class", "varchar");
        TYPE_MAP.put("java.util.Date", "date");
        TYPE_MAP.put("java.util.Calendar", "timestamp");
        TYPE_MAP.put("java.util.Locale", "varchar");
        TYPE_MAP.put("java.util.TimeZone", "varchar");
        TYPE_MAP.put("java.util.Currency", "varchar");
        TYPE_MAP.put("java.sql.Date", "date");
        TYPE_MAP.put("java.sql.Time", "time");
        TYPE_MAP.put("java.sql.Timestamp", "timestamp");
        TYPE_MAP.put("java.sql.Clob", "clob");
        TYPE_MAP.put("java.sql.Blob", "blob");
    }

    /**
     * 修改表名的SQL语句
     *
     * @param table    原表名
     * @param newTable 新表名
     * @return SQL语句
     */
    public static String renameTable(String table, String newTable) {
        return String.format("rename table %s to %s;", table, newTable);
    }

    /**
     * 增加单个字段的SQL语句
     *
     * @param table  表名
     * @param name   字段名
     * @param type   字段类型
     * @param length 字段长度
     * @return SQL语句
     */
    public static String addColumn(String table, String name, String type, Integer length) {
        type = TYPE_MAP.get(type);
        boolean hasValue = !Strings.isNullOrEmpty(table)
                && !Strings.isNullOrEmpty(name)
                && nonNull(type);
        StringBuilder result = new StringBuilder();
        if (hasValue) {
            //基本部分
            result.append(String.format("ALTER TABLE %s ADD COLUMN %s %s", table, name, type));
            //字段长度
            if (nonNull(length)) {
                result.append(String.format("(%s)", length));
            }
            result.append(";");
            return result.toString();
        } else {
            log.error("表名或字段名不能为空");
            throw new RuntimeException("表名或字段名不能为空");
        }
    }

    /**
     * 修改表的单个字段的SQL语句
     *
     * @param table     表名
     * @param name      旧字段名
     * @param newName   新字段名
     * @param newType   字段类型
     * @param newLength 字段长度
     * @return SQL语句
     */
    public static String updateColumn(String table, String name, String newName, String newType, Integer newLength) {
        newType = TYPE_MAP.get(newType);
        boolean hasValue = !Strings.isNullOrEmpty(table)
                && nonNull(newType);
        StringBuilder result = new StringBuilder();
        if (hasValue) {
            if (isNull(newName)) {
                result.append(String.format("ALTER TABLE %s MODIFY %s %s", table, name, newType));
            } else {
                result.append(String.format("ALTER TABLE %s CHANGE COLUMN %s %s %s", table, name, newName, newType));
            }
            if (nonNull(newLength)) {
                result.append(String.format("(%s)", newLength));
            }
        } else {
            log.error("参数缺失");
        }
        result.append(";");
        return result.toString();
    }

    /**
     * 删除单个字段的SQL语句
     *
     * @param table 表名
     * @param name  字段名
     * @return SQL语句
     */
    public static String deleteColumn(String table, String name) {
        return String.format("ALTER TABLE %s DROP %s;", table, name);
    }

}
