package com.tykj.workflowcore.user.controller;

import com.google.gson.JsonObject;
import com.sun.istack.NotNull;
import com.tykj.workflowcore.user.dao.LogContentDao;
import com.tykj.workflowcore.user.pojo.LogContent;
import com.tykj.workflowcore.user.pojo.User;
import com.tykj.workflowcore.user.pojo.vo.ServerResponse;
import com.tykj.workflowcore.user.pojo.vo.UserSelectVo;
import com.tykj.workflowcore.user.pojo.vo.uservo.*;
import com.tykj.workflowcore.user.service.UserService;
import com.tykj.workflowcore.user.util.UUIDUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@CrossOrigin
@Controller
@RestController()
@RequestMapping("/centerUser")
@Api(tags = "用户管理")
public class CenterUserController {
    @Autowired
    private UserService userService;
    @Autowired
    private LogContentDao logContentDao;
    @Autowired
    PasswordEncoder passwordEncoder;

    @ApiOperation("新增用户")
    @PostMapping("/addUser")
    public ServerResponse addUser(@Validated @RequestBody AddUserVo addUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(addUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("新增用户");
        User user = addUserVo.getUser();
        user.setPassword(passwordEncoder.encode(user.getPassword()));
        user.setId(UUIDUtil.getUUID());
        System.out.println("新增用户");
        if (userService.addUser(user) !=null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);
            return ServerResponse.ok(user);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);
        return ServerResponse.error("新增用户失败");
    }


    @ApiOperation("根据用户id 查看用户详情")
    @PostMapping("/findById")
    public ServerResponse findUserById(@Validated @RequestBody FindUserVo findUserVo){
        User user = userService.findById(findUserVo.getId());
        if (user != null){
            System.out.println(user.toString());
            return ServerResponse.ok(user);
        }
        return ServerResponse.error("未查到该用户");
    }


    @ApiOperation("修改用户信息")
    @PostMapping("/update")
    public ServerResponse update(@Validated @RequestBody AddUserVo addUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(addUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("修改用户详情");
        User user1 = userService.update(addUserVo.getUser());
        if (user1!=null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);
            return ServerResponse.ok(user1);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);
        return ServerResponse.error("修改用户信息失败！");
    }


    @ApiOperation("查询用户列表 新增时间倒叙排列，可按姓名进行模糊查询")
    @PostMapping("/userList")
    public ServerResponse userList(@Validated @RequestBody UserSelectVo userSelectVo){

        List<User> userList = userService.findByNameLike(userSelectVo.getRealName());
        if (userList != null){
            return ServerResponse.ok(userList);
        }
        return ServerResponse.error("未查到用户信息");
    }



    @ApiOperation("用户解锁")
    @PostMapping("/unlock")
    public ServerResponse unlock(@Validated @RequestBody FindUserVo findUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(findUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("用户解锁");

        Integer unlock = userService.unlock(findUserVo.getId());
        if (unlock != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);

            return ServerResponse.ok(unlock);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);

        return ServerResponse.error("用户解锁失败");
    }

    @ApiOperation("用户停用")
    @PostMapping("/lock")
    public ServerResponse lock(@Validated @RequestBody FindUserVo findUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(findUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("用户停用");

        Integer unlock = userService.locked(findUserVo.getId());
        if (unlock != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);

            return ServerResponse.ok(unlock);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);
        return ServerResponse.error("用户停用");
    }




    @ApiOperation("重置用户密码")
    @PostMapping("/reset")
    public ServerResponse resetPassword(@Validated @RequestBody FindUserVo findUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(findUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("重置用户密码");

        User user = userService.resetPassword(findUserVo.getId());
        if (user != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);

            return ServerResponse.ok(user);
        }

        logContent.setResult("操作失败");
        logContentDao.save(logContent);
        return ServerResponse.error("重置密码失败");
    }

    @ApiOperation("用户修改密码")
    @PostMapping("/changePassword")
    public ServerResponse changePassword(@Validated @RequestBody ChangePasswordVo changePasswordVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(changePasswordVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("修改用户密码");

        User user = userService.findById(changePasswordVo.getId());
        System.out.println(changePasswordVo.getPassword());
        String password = changePasswordVo.getPassword();
        user.setPassword(passwordEncoder.encode(password));
        User user1 = userService.update(user);
        if (user1 != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);

            return ServerResponse.ok(user1);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);

        return ServerResponse.error("修改密码失败");
    }

    @ApiOperation("删除用户")
    @DeleteMapping("/delete")
    public ServerResponse delete(@NotNull @RequestBody FindUserVo findUserVo){
        LogContent logContent = new LogContent();
        logContent.setOperator(findUserVo.getLoginer());
        logContent.setOperateTime(new Date());
        logContent.setContent("删除用户");

        User user = userService.delete(findUserVo.getId());
        if (user != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);
            return ServerResponse.ok(user);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);

        return ServerResponse.error("删除用户失败");
    }

    @ApiOperation("上传用户头像")
    @PostMapping("/upload")
    public ServerResponse uploadImage(@NotNull @RequestParam("file") MultipartFile file,@NotNull @RequestParam String loginer) throws IOException {
        LogContent logContent = new LogContent();
        logContent.setOperator(loginer);
        logContent.setOperateTime(new Date());
        logContent.setContent("上传用户头像");

        String imagePath = userService.uploadImage(file,loginer);

        if (imagePath != null){
            logContent.setResult("操作成功");
            logContentDao.save(logContent);

            return ServerResponse.ok(imagePath);
        }
        logContent.setResult("操作失败");
        logContentDao.save(logContent);

        return ServerResponse.error("上传头像失败");
    }


    @ApiOperation("根据密级展示所有用户")
    @GetMapping("/findAllByLever/{loginer}")
    public ServerResponse findAllByLever(@NotNull @PathVariable String loginer) {
        List<UserLeverVo> userLeverVos = new ArrayList<>();

        Level[] values = Level.values();
        for (Level value : values) {
            UserLeverVo userLeverVo = new UserLeverVo();
            List<User> userList = userService.findByCustomerLeverAndDelStatus(value.getName());
            userLeverVo.setCustomerLever(value.getName());
            userLeverVo.setUserList(userList);
            userLeverVos.add(userLeverVo);
        }
        if (userLeverVos!=null){
            return ServerResponse.ok(userLeverVos);
        }
        return ServerResponse.error("未查到该密级的用户");
    }


    @ApiOperation("查看所有用户")
    @GetMapping("/findAll/{loginer}")
    public ServerResponse findAll(@NotNull @PathVariable String loginer) {
        List<User> userList = userService.findAll();
        if (userList != null){
            return ServerResponse.ok(userList);
        }
        return ServerResponse.error("查看所有用户失败");
    }


    @ApiOperation("根据用户真实姓名和密级模糊查询")
    @PostMapping("/findRealname")
    public ServerResponse findRealname(@Validated @RequestBody RealNameVo realNameVo){
        List<User> userList = userService.findByRealName(realNameVo.getRealName());
        if (userList.size()==0){
            return ServerResponse.ok(new ArrayList<>() );
        }
        List<UserLeverVo> userLeverVos = new ArrayList<>();
        Level[] values = Level.values();
        for (Level value : values) {
            UserLeverVo userLeverVo = new UserLeverVo();
            ArrayList<User> users = new ArrayList<>();
            for (User user : userList) {
                if (user.getCustomerLever()!=null && user.getCustomerLever().equals(value.getName())){
                    users.add(user);
                }
            }
            userLeverVo.setCustomerLever(value.getName());
            userLeverVo.setUserList(users);
            userLeverVos.add(userLeverVo);
        }
        if (userLeverVos!=null){
            return ServerResponse.ok(userLeverVos);
        }
        return ServerResponse.error("未查到该密级的用户");
    }



}
