package com.tykj.workflowcore.model.service;

import com.tykj.workflowcore.model.entity.Bind;
import com.tykj.workflowcore.model.entity.TableInfo;
import com.tykj.workflowcore.model.entity.TableInfoEx;
import com.tykj.workflowcore.model.repository.BindRepository;
import com.tykj.workflowcore.model.repository.TableInfoExRepository;
import com.tykj.workflowcore.model.repository.TableInfoRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static java.util.Objects.nonNull;

@SuppressWarnings("Duplicates")
@Service
public class TableInfoExService {

    @Autowired
    TableInfoExRepository tableInfoExRepository;
    @Autowired
    TableInfoRepository tableInfoRepository;
    @Autowired
    BindRepository bindRepository;

    public void save(TableInfoEx tableInfoEx) {
        //数据检查
        boolean exists = tableInfoExRepository.existsByName(tableInfoEx.getName());
        if (exists) {
            throw new RuntimeException("该模型已存在");
        }
        if (nonNull(tableInfoEx.getId())) {
            throw new RuntimeException("新增操作不可附带id");
        }
        //设置时间
        Date date = new Date();
        tableInfoEx.setCreatedTime(date);
        tableInfoEx.setUpdatedTime(date);
        //设置版本号
        tableInfoEx.setVersion(1);
        //保存数据
        Integer id = tableInfoExRepository.save(tableInfoEx).getId();
        if (nonNull(tableInfoEx.getBinds())) {
            tableInfoEx.getBinds().forEach(bind -> save(bind, id));
        }
    }

    public void update(TableInfoEx tableInfoEx) {
        //数据检查
        boolean exists = tableInfoExRepository.existsByName(tableInfoEx.getName());
        if (!exists) {
            throw new RuntimeException("该模型不存在");
        }
        if (nonNull(tableInfoEx.getId())) {
            throw new RuntimeException("修改操作不可附带id");
        }
        Integer count = tableInfoExRepository.countByName(tableInfoEx.getName());
        //设置时间
        Date date = new Date();
        tableInfoEx.setCreatedTime(date);
        tableInfoEx.setUpdatedTime(date);
        tableInfoEx.setVersion(count + 1);
        //保存数据
        Integer id = tableInfoExRepository.save(tableInfoEx).getId();
        if (nonNull(tableInfoEx.getBinds())) {
            tableInfoEx.getBinds().forEach(bind -> save(bind, id));
        }
    }


    public List<TableInfoEx> findAll() {
        return tableInfoExRepository.findAll().stream()
                .map(this::getBinds)
                .collect(Collectors.toList());
    }

    public void updateRelatedTableInfoEx(TableInfo tableInfo) {
        List<String> names = bindRepository.findByName(tableInfo.getName()).stream()
                .map(Bind::getTableInfoExId)
                .distinct()
                .map(tableInfoExId -> tableInfoExRepository.findById(tableInfoExId).orElse(null))
                .filter(Objects::nonNull)
                .map(TableInfoEx::getName)
                .distinct()
                .collect(Collectors.toList());
        for (String name : names) {
            TableInfoEx tableInfoEx = tableInfoExRepository.findByName(name).stream()
                    .max(Comparator.comparingInt(TableInfoEx::getVersion))
                    .map(this::getBinds)
                    .orElseThrow(() -> new RuntimeException("计算中出现异常"));
            Integer lastVersion = tableInfoEx.getVersion();
            List<Bind> binds = tableInfoEx.getBinds();
            TableInfoEx newTableInfoEx = new TableInfoEx()
                    .setName(tableInfoEx.getName())
                    .setDescription(tableInfoEx.getDescription())
                    .setProcessKey(tableInfoEx.getProcessKey())
                    .setVersion(lastVersion + 1)
                    .setBinds(newBinds(binds, tableInfo));
            update(newTableInfoEx);
        }
    }

    //-----------------------------------------------------------------------------//

    private List<Bind> newBinds(List<Bind> binds, TableInfo tableInfo) {
        List<Bind> newBinds = new ArrayList<>();
        for (Bind bind : binds) {
            Bind newBind = new Bind()
                    .setName(bind.getName())
                    .setVersion(bind.getVersion())
                    .setIsComplex(bind.getIsComplex());
            if (Objects.equals(bind.getName(), tableInfo.getName())) {
                newBind.setVersion(tableInfo.getVersion());
            }
            if (hasChildren(bind)) {
                List<Bind> children = bind.getChildren();
                newBind.setChildren(newBinds(children, tableInfo));
            }
            newBinds.add(newBind);
        }
        return newBinds;
    }

    private boolean hasChildren(Bind bind) {
        return nonNull(bind.getChildren()) && !bind.getChildren().isEmpty();
    }

    private void save(Bind bind, Integer tableInfoExId) {
        Integer lastVersion = tableInfoRepository.countByName(bind.getName());
        bind.setVersion(lastVersion);
        bind.setTableInfoExId(tableInfoExId);
        Integer id = bindRepository.save(bind).getId();
        if (hasChildren(bind)) {
            List<Bind> childrenForSave = bind.getChildren().stream()
                    .map(child -> child.setParentId(id))
                    .collect(Collectors.toList());
            childrenForSave.forEach(child -> save(child, tableInfoExId));
        }
    }

    private TableInfoEx getBinds(TableInfoEx tableInfoEx) {
        List<Bind> binds = bindRepository.findByTableInfoExIdAndParentId(tableInfoEx.getId(), null).stream()
                .map(this::getChildren)
                .collect(Collectors.toList());
        tableInfoEx.setBinds(binds);
        return tableInfoEx;
    }

    private Bind getChildren(Bind bind) {
        List<Bind> children = bindRepository.findByTableInfoExIdAndParentId(bind.getTableInfoExId(), bind.getId()).stream()
                .map(child -> child.setChildren(Collections.emptyList()))
                .collect(Collectors.toList());
        bind.setChildren(children);
        return bind;
    }

}
