package com.tykj.workflowcore.model.service;

import com.tykj.workflowcore.model.entity.ColumnInfo;
import com.tykj.workflowcore.model.entity.TableInfo;
import com.tykj.workflowcore.model.repository.ColumnInfoRepository;
import com.tykj.workflowcore.model.repository.TableInfoRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;


import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import static java.util.Objects.nonNull;

@SuppressWarnings("Duplicates")
@Service
public class TableInfoService {

    @Autowired
    private TableInfoExService tableInfoExService;
    @Autowired
    private TableInfoRepository tableInfoRepository;
    @Autowired
    private ColumnInfoRepository columnInfoRepository;

    public void save(TableInfo tableInfo) {
        //数据检查
        boolean exists = tableInfoRepository.existsByName(tableInfo.getName());
        if (exists) {
            throw new RuntimeException("该模型已存在");
        }
        if (nonNull(tableInfo.getId())) {
            throw new RuntimeException("新增操作不可附带id");
        }
        //设置时间
        Date date = new Date();
        tableInfo.setCreatedTime(date);
        tableInfo.setUpdatedTime(date);
        //设置版本号
        tableInfo.setVersion(1);
        //保存数据
        Integer id = tableInfoRepository.save(tableInfo).getId();
        List<ColumnInfo> columnInfosForSave = tableInfo.getColumnInfos().stream()
                .map(columnInfo -> columnInfo.setTableInfoId(id))
                .collect(Collectors.toList());
        columnInfosForSave.forEach(columnInfoRepository::save);
    }

    public void update(TableInfo tableInfo) {
        //数据检查
        boolean exists = tableInfoRepository.existsByName(tableInfo.getName());
        if (!exists) {
            throw new RuntimeException("该模型不存在");
        }
        if (nonNull(tableInfo.getId())) {
            throw new RuntimeException("修改操作不可附带id");
        }
        Integer count = tableInfoRepository.countByName(tableInfo.getName());
        //设置时间
        Date date = new Date();
        tableInfo.setCreatedTime(date);
        tableInfo.setUpdatedTime(date);
        tableInfo.setVersion(count + 1);
        //保存数据
        Integer id = tableInfoRepository.save(tableInfo).getId();
        List<ColumnInfo> columnInfosForSave = tableInfo.getColumnInfos().stream()
                .map(columnInfo -> columnInfo.setTableInfoId(id))
                .collect(Collectors.toList());
        columnInfosForSave.forEach(columnInfoRepository::save);
        //更新有关联的聚合对象
        tableInfoExService.updateRelatedTableInfoEx(tableInfo);
    }

    public TableInfo findById(Integer id) {
        TableInfo tableInfo = tableInfoRepository.findById(id).orElseThrow(() -> new RuntimeException("该模型不存在"));
        List<ColumnInfo> columnInfos = columnInfoRepository.findByTableInfoId(id);
        tableInfo.setColumnInfos(columnInfos);
        return tableInfo;
    }

    public List<TableInfo> findAll() {
        return tableInfoRepository.findAll().stream()
                .map(this::getColumnInfos)
                .collect(Collectors.toList());
    }

    //-----------------------------------------------------------------------------//

    private TableInfo getColumnInfos(TableInfo tableInfo) {
        List<ColumnInfo> columnInfos = columnInfoRepository.findByTableInfoId(tableInfo.getId());
        tableInfo.setColumnInfos(columnInfos);
        return tableInfo;
    }

}
