package com.zjty.adaptationmaster.adaptor.examination;

import com.zjty.adaptationmaster.adaptor.enginer.util.StringCompareUtil;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告
 *
 */
public class Inspectors {
    private String path;

    private Map<String, Counter> matchMap;
    private List<Path> propertiesConfigPaths = new ArrayList<>();
    private List<Path> ymlConfigPaths = new ArrayList<>();
    private List<Path> xmlConfigPaths = new ArrayList<>();

    private Report report = new Report();

    public void setPath(String path) {
        this.path = path;
    }

    public void setMatchMap(Map<String, Counter> matchMap) {
        this.matchMap = matchMap;
    }

    public static void main(String[] args) {
        Inspectors inspectors = new Inspectors();
        inspectors.setPath("C:\\Users\\wyl\\Desktop\\testReplace");

        Map<String,Counter> map = new HashMap<>();
        map.put("java",new Counter());

        map.put("jsp",new Counter());
        map.put("html",new Counter());

        inspectors.setMatchMap(map);

        inspectors.inspect();
        //打印分析数据
        for(Map.Entry<String,Counter> entry:map.entrySet()) {
            System.out.println("项目中"+entry.getKey()+"类型文件共有" + entry.getValue().getI()+"个");
        }
        //打印体检报告
        System.out.println(inspectors.analysis());
    }

    /**
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     */
    public void inspect(){
        try {
            String glob = "glob:**/*.txt";
            final PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher( glob);
            Files.walkFileTree(Paths.get(path), new FileVisitor<Path>() {
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    //判断后缀
                    for (Map.Entry<String, Counter> s : matchMap.entrySet()) {
                        if (dir.endsWith(s.getKey())) {
                            s.getValue().plus();
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    boolean matches = pathMatcher.matches(file);
                    if (file.endsWith("properties")) {
                        propertiesConfigPaths.add(file);
                    }
                    if(file.endsWith("yml")){
                        ymlConfigPaths.add(file);
                    }
                    if(file.endsWith("xml")){
                        xmlConfigPaths.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public Report analysis(){
        if(matchMap.get("java").getI()>0){

            report.setType("java");
        }
        if(matchMap.get("jsp").getI()>0){
            report.setStructure("whole");
        }
        if(matchMap.get("html").getI()>0){
            report.setStructure("separation");
        }
        for(Path path:propertiesConfigPaths){
            try {
                for(String s:Files.readAllLines(path)){
                    int index = StringCompareUtil.compare(s, "spring.datasource.driver-class-name");
                    if(index >- 1){
                        if(!s.startsWith("#")){
                            /**
                             * MySQL数据库：
                             *
                             *     1）驱动包：https://mvnrepository.com/artifact/mysql/mysql-connector-java（下载路径）
                             *
                             *     2）驱动类名：com.mysql.jdbc.Driver
                             *
                             *     3）JDBC的URL：jdbc:mysql://IP地址:端口号/数据库名字
                             *
                             *      注：端口号缺省为：3306
                             *
                             * SQL server数据库：
                             *
                             *     1）驱动包：https://mvnrepository.com/artifact/com.microsoft.sqlserver/sqljdbc4（下载路径）
                             *
                             *     2）驱动类名：com.microsoft.jdbc.sqlserver.SQLServerDriver
                             *
                             *     3）JDBC的URL：jdbc:microsoft:sqlserver://IP地址:端口号;DatabaseName=数据库名
                             *
                             *     4）sqljdbc和sqljdbc4区别：https://blog.csdn.net/cainiao_M/article/details/53404222
                             *
                             *     注：端口号缺省为：1433
                             *
                             * Oracle数据库：
                             *
                             *     1）驱动包：https://mvnrepository.com/artifact/com.oracle/ojdbc6（下载路径）
                             *
                             *     2）驱动类名：oracle.jdbc.driver.OracleDriver
                             *
                             *     3）JDBC的URL：jdbc:oracle:thin:@IP地址:端口号:数据库名
                             *
                             *     4）ojdbc6和ojdbc14的区别：ojdbc14.jar（适合java-1.4和1.5），ojdbc6（适合java-1.6）
                             *
                             *     注：端口号缺省为：1521
                             */
                            if(StringCompareUtil.compare(s, "mysql")>-1){
                                report.setSqlType("mysql");
                            }else if(StringCompareUtil.compare(s, "sqlserver")>-1){
                                report.setSqlType("SQLServer");
                            }else if(StringCompareUtil.compare(s, "oracle")>-1){
                                report.setSqlType("oracle");
                            }else if(StringCompareUtil.compare(s, "postgre")>-1){
                                report.setSqlType("postgre");
                            }
                        }
                    }
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return null;
    }
    private static class Counter {
        private int i = 0;
        public void plus(){
            i++;
        }

        public int getI() {
            return i;
        }

        public void reset(){
            i = 0;
        }
    }
}
