package com.zjty.adaptationmaster.utils;

import com.zjty.adaptationmaster.adaptor.entity.Project;
import com.zjty.adaptationmaster.adaptor.entity.Report;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告
 *
 */
public class Inspector {

    public static void main(String[] args) {
        Project project = new Project();
        project.setCodeUrl("C:\\home\\project\\rsc\\hrmbclient");
        Map<String, Report.Language> suffixLanguageMapping = new HashMap<>();
        suffixLanguageMapping.put("java",Report.Language.JAVA);
        suffixLanguageMapping.put("cpp",Report.Language.CPP);
        suffixLanguageMapping.put("py",Report.Language.PYTHON);
        suffixLanguageMapping.put("jsp",Report.Language.JSP);
        Inspector inspector = new Inspector(project,suffixLanguageMapping);
        System.out.println(inspector.inspect());
    }

    private Project project;
    private Report report;
    private Map<String, Counter> languageMatchMap;
    private Map<String, Report.Language> suffixLanguageMapping;

    private Map<String,List<Path>> configFileTypePathsMapping;

    public Inspector(Project project,Map<String,Report.Language> suffixLanguageMapping){
        this.project = project;
        this.suffixLanguageMapping = suffixLanguageMapping;
        this.languageMatchMap = new HashMap<>();
        for(String s:suffixLanguageMapping.keySet()){
            languageMatchMap.put(s,new Counter());
        }
        /**
         * 如果存在html文件，不一定是前端代码，
         * 如果只存在html文件，不存在其他类型文件，大概率是前端代码
         * 策略：
         *      先检测是否存在html文件，如果存在，假定为前端项目
         *      然后检测其他条件，如果符合，将假定冲掉
         */
        languageMatchMap.put("html",new Counter());
        this.configFileTypePathsMapping = new HashMap<>();
        configFileTypePathsMapping.put("xml",new ArrayList<>());
        configFileTypePathsMapping.put("properties",new ArrayList<>());
        configFileTypePathsMapping.put("yml",new ArrayList<>());
        this.report = new Report();
    }

    /**
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     */
    public Report inspect(){
        try {
            FileSystem aDefault = FileSystems.getDefault();
            Map<String,PathMatcher> languageSuffixMatcherMapping = new HashMap<>();
            for(String s:suffixLanguageMapping.keySet()){
                languageSuffixMatcherMapping.put(s,aDefault.getPathMatcher("glob:**/*."+s));
            }
            Map<PathMatcher,String> configFileMatcherSuffixMapping = new HashMap<>();
            for(String s:configFileTypePathsMapping.keySet()){
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*."+s),s);
            }

            Files.walkFileTree(Paths.get(project.getCodeUrl()), new FileVisitor<Path>() {
                /**
                 * 统计某个后缀出现的次数
                 * @param dir
                 * @param attrs
                 * @return
                 * @throws IOException
                 */
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    /**
                     * 这里是对于路径（文件夹）的过滤，在这里读不到文件
                     * 如果能判断，可以返回FileVisitResult.SKIP_SUBTREE 不访问子目录
                     */
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    for(Map.Entry<String,PathMatcher> entry:languageSuffixMatcherMapping.entrySet()){
                        if(entry.getValue().matches(file)){
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for(Map.Entry<PathMatcher,String> entry:configFileMatcherSuffixMapping.entrySet()){
                        if(entry.getKey().matches(file)){
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    //System.out.println(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        return analysis();
    }

    public Report analysis(){
        report.setLanguage(Report.Language.ONLYVIEW);
        for(Map.Entry<String,Counter> entry:languageMatchMap.entrySet()){
            if(entry.getValue().getNumber()>0){
                if(!entry.getKey().equals("html")){
                    report.setLanguage(suffixLanguageMapping.get(entry.getKey()));
                }
            }
        }
        /**
         * 对于具体的配置文件对应的处理方法
         * 增加要处理的文件类型需要在这里增加相应的处理方法
         */
        for(Map.Entry<String,List<Path>> entry:configFileTypePathsMapping.entrySet()){
            //System.out.println(entry.getKey());
            switch (entry.getKey()){
                /**
                 * 配置文件的一个类型，xml文件
                 */
                case "xml":
                    System.out.println(entry.getValue().size());
                    for(Path path:entry.getValue()){
                        /**
                         * 对于maven工程，可以在maven配置文件中得到一这些信息
                         *      编译方式打包方式和打好的包的路径
                         */
                        if(path.getFileName().endsWith("pom.xml")){
                            report.setDependenceManagement(Report.DependenceManagement.MAVEN);
                            report.setCompileFilePath(path.toString());

                            SAXReader reader = new SAXReader();
                            Document document = null;
                            try {
                                document = reader.read(path.toFile());
                            } catch (DocumentException e) {
                                e.printStackTrace();
                            }
                            Element elementProject = document.getRootElement();
                            String projectMavenName = elementProject.elementText("artifactId");
                            String projectMavenVersion = elementProject.elementText("version");
                            report.setPackagePath(path.getParent()+"/target/"+projectMavenName+"-"+projectMavenVersion+".war");
                        }
                    }
                    break;
                /**
                 * 配置文件的一个类型，properties文件
                 *
                 */
                case "properties":
                    for(Path path:entry.getValue()) {
                        /**
                         * springboot项目的配置文件一般是application.properties或application.yml
                         */
                        if (path.getFileName().endsWith("application.properties")) {
                            Properties properties = new Properties();
                            try {
                                properties.load(new FileInputStream(path.toFile()));
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                            String datasourceDriver = properties.getProperty("spring.datasource.driver-class-name");
                            String active = properties.getProperty("spring.profiles.active");
                            System.out.println(path.getParent());
                            File file = new File(path.getParent().toString() + "/application-" + active + ".properties");
                            if (file.exists()) {
                                Properties properties1 = new Properties();
                                try {
                                    properties1.load(new FileInputStream(file));
                                    String driver = properties1.getProperty("spring.datasource.driver-class-name");
                                    if (driver != null) datasourceDriver = driver;
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }
                            } else {
                                System.out.println("没有找到active配置文件");
                            }
                            if (datasourceDriver != null) {
                                if (datasourceDriver.contains(Report.DatabaseType.MYSQL.name().toLowerCase())) {
                                    report.setDatabaseType(Report.DatabaseType.MYSQL);
                                } else if (datasourceDriver.contains(Report.DatabaseType.POSTGRE.name().toLowerCase())) {
                                    report.setDatabaseType(Report.DatabaseType.POSTGRE);
                                } else if (datasourceDriver.contains(Report.DatabaseType.ORACLE.name().toLowerCase())) {
                                    report.setDatabaseType(Report.DatabaseType.ORACLE);
                                } else if (datasourceDriver.contains(Report.DatabaseType.SQLSERVER.name().toLowerCase())) {
                                    report.setDatabaseType(Report.DatabaseType.SQLSERVER);
                                }
                            }
                        }
                    }
                    break;
                case "yml":
                    break;
            }
        }
//        for(Path path:propertiesConfigPaths){
//            try {
//                for(String s:Files.readAllLines(path)){
//                    int index = StringCompareUtil.compare(s, "spring.datasource.driver-class-name");
//                    if(index > -1){
//                        if(!s.startsWith("#")){
//                            /**
//                             * MySQL数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/mysql/mysql-connector-java（下载路径）
//                             *     2）驱动类名：com.mysql.jdbc.Driver
//                             *     3）JDBC的URL：jdbc:mysql://IP地址:端口号/数据库名字
//                             *      注：端口号缺省为：3306
//                             *
//                             * SQL server数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/com.microsoft.sqlserver/sqljdbc4（下载路径）
//                             *     2）驱动类名：com.microsoft.jdbc.sqlserver.SQLServerDriver
//                             *     3）JDBC的URL：jdbc:microsoft:sqlserver://IP地址:端口号;DatabaseName=数据库名
//                             *     4）sqljdbc和sqljdbc4区别：https://blog.csdn.net/cainiao_M/article/details/53404222
//                             *     注：端口号缺省为：1433
//                             *
//                             * Oracle数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/com.oracle/ojdbc6（下载路径）
//                             *     2）驱动类名：oracle.jdbc.driver.OracleDriver
//                             *     3）JDBC的URL：jdbc:oracle:thin:@IP地址:端口号:数据库名
//                             *     4）ojdbc6和ojdbc14的区别：ojdbc14.jar（适合java-1.4和1.5），ojdbc6（适合java-1.6）
//                             *     注：端口号缺省为：1521
//                             */
//                            if(StringCompareUtil.compare(s, "mysql")>-1){
//                                report.setDatabaseType(Report.DatabaseType.MYSQL);
//                            }else if(StringCompareUtil.compare(s, "sqlserver")>-1){
//                                report.setDatabaseType(Report.DatabaseType.SQLSERVER);
//                            }else if(StringCompareUtil.compare(s, "oracle")>-1){
//                                report.setDatabaseType(Report.DatabaseType.ORACLE);
//                            }else if(StringCompareUtil.compare(s, "postgre")>-1){
//                                report.setDatabaseType(Report.DatabaseType.POSTGRE);
//                            }
//                        }
//                    }
//                }
//            } catch (IOException e) {
//                e.printStackTrace();
//            }
//        }
        return report;
    }
    public class Counter {
        private int i = 0;
        public void plus(){
            i++;
        }

        public int getNumber() {
            return i;
        }

        public void reset(){
            i = 0;
        }
    }
}
