package com.zjty.adaptationmaster.utils;

import com.zjty.adaptationmaster.adaptor.entity.Project;
import com.zjty.adaptationmaster.adaptor.entity.Report;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告
 *
 */
public class Inspector {

    public static void main(String[] args) {
        Project project = new Project();
        project.setCodeUrl("");
        Map<String,Report.Language> suffixLanguageMapping = new HashMap<>();
        Map<String, Report.Separate> suffixSeparateMapping = new HashMap<>();
        List<String> configFileSuffixList = new ArrayList<>();
        Inspector inspector = new Inspector(project,suffixLanguageMapping,suffixSeparateMapping,configFileSuffixList);
    }

    private Project project;
    private Report report;
    private Map<String, Counter> languageMatchMap;
    private Map<String, Counter> separateMatchMap;
    private Map<String, Report.Language> suffixLanguageMapping;
    private Map<String, Report.Separate> suffixSeparateMapping;

    private Map<String,List<Path>> configFileTypePathsMapping;

    public Inspector(Project project,Map<String,Report.Language> suffixLanguageMapping,Map<String, Report.Separate> suffixSeparateMapping,List<String> configFileSuffixList){
        this.project = project;
        this.suffixLanguageMapping = suffixLanguageMapping;
        this.suffixSeparateMapping = suffixSeparateMapping;
        this.languageMatchMap = new HashMap<>();
        for(String s:suffixLanguageMapping.keySet()){
            languageMatchMap.put(s,new Counter());
        }
        this.separateMatchMap = new HashMap<>();
        for(String s:suffixSeparateMapping.keySet()){
            separateMatchMap.put(s,new Counter());
        }
        this.configFileTypePathsMapping = new HashMap<>();
        for(String s:configFileSuffixList){
            configFileTypePathsMapping.put(s,new ArrayList<>());
        }
        this.report = new Report();
    }

    private List<Path> propertiesConfigPaths = new ArrayList<>();
    private List<Path> ymlConfigPaths = new ArrayList<>();
    private List<Path> xmlConfigPaths = new ArrayList<>();

    /**
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     */
    public Report inspect(){
        try {
            FileSystem aDefault = FileSystems.getDefault();
            Map<String,PathMatcher> languageSuffixMatcherMapping = new HashMap<>();
            for(String s:suffixLanguageMapping.keySet()){
                languageSuffixMatcherMapping.put(s,aDefault.getPathMatcher("glob:**/*."+s));
            }
            Map<String,PathMatcher> separateSuffixMatcherMapping = new HashMap<>();
            for(String s:suffixLanguageMapping.keySet()){
                separateSuffixMatcherMapping.put(s,aDefault.getPathMatcher("glob:**/*."+s));
            }

            Map<PathMatcher,String> configFileMatcherSuffixMapping = new HashMap<>();
            for(String s:configFileTypePathsMapping.keySet()){
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*."+s),s);
            }

            Files.walkFileTree(Paths.get(project.getCodeUrl()), new FileVisitor<Path>() {
                /**
                 * 统计某个后缀出现的次数
                 * @param dir
                 * @param attrs
                 * @return
                 * @throws IOException
                 */
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    for(Map.Entry<String,PathMatcher> entry:languageSuffixMatcherMapping.entrySet()){
                        if(entry.getValue().matches(dir)){
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for(Map.Entry<String,PathMatcher> entry:separateSuffixMatcherMapping.entrySet()){
                        if(entry.getValue().matches(dir)){
                            separateMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    for(Map.Entry<PathMatcher,String> entry:configFileMatcherSuffixMapping.entrySet()){
                        if(entry.getKey().matches(file)){
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        return analysis();
    }

    public Report analysis(){
        for(Map.Entry<String,Counter> entry:languageMatchMap.entrySet()){
            if(entry.getValue().getNumber()>0){
                report.setLanguage(suffixLanguageMapping.get(entry.getKey()));
            }
        }
        for(Map.Entry<String,Counter> entry:separateMatchMap.entrySet()){
            if(entry.getValue().getNumber()>0){
                report.setIsSeparate(suffixSeparateMapping.get(entry.getKey()));
            }
        }
        for(Map.Entry<String,List<Path>> entry:configFileTypePathsMapping.entrySet()){

        }
//        for(Path path:propertiesConfigPaths){
//            try {
//                for(String s:Files.readAllLines(path)){
//                    int index = StringCompareUtil.compare(s, "spring.datasource.driver-class-name");
//                    if(index > -1){
//                        if(!s.startsWith("#")){
//                            /**
//                             * MySQL数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/mysql/mysql-connector-java（下载路径）
//                             *     2）驱动类名：com.mysql.jdbc.Driver
//                             *     3）JDBC的URL：jdbc:mysql://IP地址:端口号/数据库名字
//                             *      注：端口号缺省为：3306
//                             *
//                             * SQL server数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/com.microsoft.sqlserver/sqljdbc4（下载路径）
//                             *     2）驱动类名：com.microsoft.jdbc.sqlserver.SQLServerDriver
//                             *     3）JDBC的URL：jdbc:microsoft:sqlserver://IP地址:端口号;DatabaseName=数据库名
//                             *     4）sqljdbc和sqljdbc4区别：https://blog.csdn.net/cainiao_M/article/details/53404222
//                             *     注：端口号缺省为：1433
//                             *
//                             * Oracle数据库：
//                             *     1）驱动包：https://mvnrepository.com/artifact/com.oracle/ojdbc6（下载路径）
//                             *     2）驱动类名：oracle.jdbc.driver.OracleDriver
//                             *     3）JDBC的URL：jdbc:oracle:thin:@IP地址:端口号:数据库名
//                             *     4）ojdbc6和ojdbc14的区别：ojdbc14.jar（适合java-1.4和1.5），ojdbc6（适合java-1.6）
//                             *     注：端口号缺省为：1521
//                             */
//                            if(StringCompareUtil.compare(s, "mysql")>-1){
//                                report.setDatabaseType(Report.DatabaseType.MYSQL);
//                            }else if(StringCompareUtil.compare(s, "sqlserver")>-1){
//                                report.setDatabaseType(Report.DatabaseType.SQLSERVER);
//                            }else if(StringCompareUtil.compare(s, "oracle")>-1){
//                                report.setDatabaseType(Report.DatabaseType.ORACLE);
//                            }else if(StringCompareUtil.compare(s, "postgre")>-1){
//                                report.setDatabaseType(Report.DatabaseType.POSTGRE);
//                            }
//                        }
//                    }
//                }
//            } catch (IOException e) {
//                e.printStackTrace();
//            }
//        }
        return null;
    }
    public class Counter {
        private int i = 0;
        public void plus(){
            i++;
        }

        public int getNumber() {
            return i;
        }

        public void reset(){
            i = 0;
        }
    }
}
