package com.zjty.inspect.utils;

import com.zjty.inspect.dao.BudgetDao;
import com.zjty.inspect.dao.RuleDao;
import com.zjty.inspect.dao.TechnologyAndReportDao;
import com.zjty.inspect.dao.TechnologyDao;
import com.zjty.inspect.entity.*;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告VO
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
@Service
@Scope("prototype")
public class Inspector {

//    @Autowired
//    private RedisUtil redisUtil;
    @Autowired
    private AnalysisFile analysisFile;
    @Autowired
    private MavenUtil mavenUtil;
    @Autowired
    BudgetUitl budgetUitl;

    @Autowired
    private TechnologyAndReportDao technologyAndReportDao;
    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyDao technologyDao;
    @Autowired
    private BudgetDao budgetDao;
    private DepTreeVo depTreeVo = new DepTreeVo();

    private ArrayList<TechnologyAndReport> technologyAndReports = new ArrayList<>();
    /**
     * 利率参数
     */
    private InspectParameter inspectParameter;
    /**
     * 报告对象
     */
    private Report report;

    /**
     * 语言
     */
    private Map<String, Counter> languageMatchMap = new HashMap<>();

    /**
     * 后缀语言
     */
    private Map<String, Report.Language> suffixLanguageMapping = new HashMap<>();
    /**
     * 路径
     */
    private String path;

    /**
     * 规则列表
     */
    private List<Rule> ruleList;

    /**
     * 配置文件后缀
     * xml：list【路径】
     */
    private Map<String, List<Path>> configFileTypePathsMapping = new HashMap<>();

    private Map<String,List<Path>> ruleSuffixMap = new HashMap<>();
    private Map<String,List<Rule>> ruleSuffixList = new HashMap<>();
    /**
     * 统计各后缀文件路径与出现次数,顺便解析jar与js文件
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     * @return 报告
     */
    public Report inspect() {
        report.setId(UUIDUtil.getUUID());
        for (String s : suffixLanguageMapping.keySet()) {
            //配置如：java，0
            languageMatchMap.put(s, new Counter());
        }
        /**
         * 如果存在html文件，不一定是前端代码，
         * 如果只存在html文件，不存在其他类型文件，大概率是前端代码
         * 策略：
         *      先检测是否存在html文件，如果存在，假定为前端项目
         *      然后检测其他条件，如果符合，将假定冲掉
         */
        this.configFileTypePathsMapping = new HashMap<>();
        configFileTypePathsMapping.put("xml", new ArrayList<>());
        configFileTypePathsMapping.put("json", new ArrayList<>());
        configFileTypePathsMapping.put("gradle", new ArrayList<>());
        configFileTypePathsMapping.put("properties", new ArrayList<>());
        configFileTypePathsMapping.put("yml", new ArrayList<>());
        this.ruleList=ruleDao.findAll();
        for (Rule rule : ruleList) {
            if(!ruleSuffixMap.containsKey(rule.getSuffix())){
                ruleSuffixMap.put(rule.getSuffix(),new ArrayList<>());
                ruleSuffixList.put(rule.getSuffix(),new ArrayList<>());
                ruleSuffixList.get(rule.getSuffix()).add(rule);
            }else{
                ruleSuffixList.get(rule.getSuffix()).add(rule);
            }
        }

        try {
            FileSystem aDefault = FileSystems.getDefault();
            Map<String, PathMatcher> languageSuffixMatcherMapping = new HashMap<>(16);
            for (String s : suffixLanguageMapping.keySet()) {
                languageSuffixMatcherMapping.put(s, aDefault.getPathMatcher("glob:**/*." + s));
            }
            Map<PathMatcher, String> configFileMatcherSuffixMapping = new HashMap<>(16);
            for (String s : configFileTypePathsMapping.keySet()) {
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            Map<PathMatcher, String> ruleSuffix = new HashMap<>(16);
            for (String s : ruleSuffixMap.keySet()) {
                ruleSuffix.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            /// TODO: 2020-02-12
            // WebSocketServer.sendIn(uuid,TimeUtil.getNowDate() + " 项目评估：" + "开始读取项目文件");
            Files.walkFileTree(Paths.get(path), new FileVisitor<Path>() {
                /**
                 * 统计某个后缀出现的次数
                 * @param dir
                 * @param attrs
                 * @return
                 * @throws IOException
                 */
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    /**
                     * 这里是对于路径（文件夹）的过滤，在这里读不到文件
                     * 如果能判断，可以返回FileVisitResult.SKIP_SUBTREE 不访问子目录
                     */
                    //if(dir.getFileName().startsWith("."))return FileVisitResult.SKIP_SUBTREE;
                    //if (dir.endsWith(".git"))return FileVisitResult.SKIP_SUBTREE;
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    
                    for (Map.Entry<String, PathMatcher> entry : languageSuffixMatcherMapping.entrySet()) {
                        if (entry.getValue().matches(file)) {
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : configFileMatcherSuffixMapping.entrySet()) {
                        if (entry.getKey().matches(file)) {
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : ruleSuffix.entrySet()) {
                        if (entry.getKey().matches(file)) {
                            ruleSuffixMap.get(entry.getValue()).add(file);
                        }
                    }
                    //获取普通依赖jar
                    if (file.toString().endsWith(".jar")) {
                        // TODO: 2020-02-28 jar文件需要修改格式变为string
                        DepTree voli = voliFileName(file.getFileName().toString());
                        depTreeVo.getDepTreeList().add(voli);
                    }
                    //获取普通js文件
                    if (file.toString().endsWith(".js")) {
                        DepTree voli = voliFileName(file.getFileName().toString());
                        depTreeVo.getFrontend().add(voli);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        return analysis();
    }

    /**
     * 构成报告所需要到数据
     * @return Report
     */
    @Transactional
    public Report analysis() {
        DepTreeVo depTreeVo = new DepTreeVo();
        depTreeVo.setId(report.getId());
        String most = null;
        int maxnum = 0;
        for (Map.Entry<String, Counter> entry : languageMatchMap.entrySet()) {
            if (entry.getValue().getNumber() > maxnum) {
                most = suffixLanguageMapping.get(entry.getKey()).name();
                maxnum = entry.getValue().getNumber();
            }
        }
        report.setLanguage(most == null ? Report.Language.UNKNOW.name() : most);
        report.setFramework(languageMatchMap.get("jsp").i>0 ? "混合型架构" : "分离型架构");
        if(languageMatchMap.get("jsp").i==0 & languageMatchMap.get("java").i==0){
            report.setRecastMethod(2);
        }else{
            report.setRecastMethod(1);
        }
        // TODO: 2020-02-28 jsFiles, jarFiles

        for (Map.Entry<String, List<Path>> entry : configFileTypePathsMapping.entrySet()) {
            switch (entry.getKey()) {
                /**
                 * 配置文件的一个类型，xml文件
                 */
                case "xml":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("pom.xml")) {
                            // TODO: 2020-02-28 解析maven树文件，设置依赖保存到redis
                            report.setManager(Report.DependenceManagement.MAVEN.name());
                            File file = mavenUtil.genTreeFile(path.toString());
                            DepTreePom depTreePom = analysisFile.parseTreeFile(report.getId(),file.getPath(),ruleList);
                            depTreeVo.getDepTreeList().add(depTreePom.getDepTree());

                            //savetoredis
                            //redisUtil.set(report.getId(),depTreeVo);
                        }
                    }
                    break;
                case "gradle":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("build.gradle")) {
                            report.setManager(Report.DependenceManagement.GRADLE.name());
                            List<PomDependency> pomDependencies = AnalysisFile.analysisGradle(path);
                            //设置依赖
                            for (PomDependency dependency : pomDependencies) {
                                DepTree depTree = voliFileName(dependency.getGradle());
                                depTreeVo.getDepTreeList().add(depTree);
                            }
                        }
                    }
                case "json":
                    //前端依赖解析
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("build.gradle")) {
                            VueDep json = FileUtil.parseJsonFile(path.toString());
                            Map<String, String> dependencies = json.getDependencies();
                            for (String s : dependencies.keySet()) {
                                DepTree depTree = voliFileName(s);
                                depTreeVo.getFrontend().add(depTree);
                            }
                        }
                    }
                    break;
                default:
            }
        }
        //指定后缀到文件匹配关键字
        for (Map.Entry<String, List<Path>> entry : ruleSuffixMap.entrySet()) {
            String key = entry.getKey();
            List<Rule> rules = ruleSuffixList.get(key);
            for (Path path1 : entry.getValue()) {
                try {
                    List<String> strings = Files.readAllLines(path1);
                    for (int i = 0; i < strings.size(); i++) {
                            for (Rule rule : rules) {
                                int ik = KmpUtil.kmpMatch(strings.get(i), rule.getTarget());
                                if(ik>0){
                                    //rule匹配上了
                                    TechnologyAndReport technologyAndReport = new TechnologyAndReport();
                                    technologyAndReport.setFilePath(path1.toString());
                                    technologyAndReport.setLineNum(i+1);
                                    technologyAndReport.setTechnologyId(rule.getTechnologyId());
                                    technologyAndReport.setReportId(report.getId());
                                    technologyAndReport.setRuleId(rule.getId());
                                    technologyAndReport.setId(UUIDUtil.getUUID());
                                    technologyAndReports.add(technologyAndReport);
                                }
                            }
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        technologyAndReportDao.saveAll(technologyAndReports);
        Set<String> collect = technologyAndReports.stream().map(TechnologyAndReport::getTechnologyId).collect(Collectors.toSet());
        List<Technology> allById = technologyDao.findAllById(collect);
        Integer fund = 0;
        for (Technology technology : allById) {
            fund+=technology.getFund();
        }
        //计算预算，三种预算
        List<Budget> budget = budgetUitl.getBudget(report.getId(),fund, inspectParameter);
        budgetDao.saveAll(budget);
        // TODO: 2020-02-28 生成报告，返回地址 report.setsourceaddress;
        return report;
    }

    /**
     * 检查普通依赖
     * @param fileName 文件名称
     * @return deptree
     */
    private DepTree voliFileName(String fileName){
        DepTree depTree = new DepTree();
        for (Rule rule : ruleList) {
            depTree.setDepName(fileName);
            if(KmpUtil.kmpMatch(fileName,rule.getTarget())>0){
                depTree.setViolation(1);
                String technologyId = rule.getTechnologyId();
                Optional<Technology> byId = technologyDao.findById(technologyId);
                if(byId.isPresent()){
                    TechnologyAndReport technologyAndReport = new TechnologyAndReport();
                    technologyAndReport.setFilePath(fileName);
                    technologyAndReport.setLineNum(1);
                    technologyAndReport.setTechnologyId(rule.getTechnologyId());
                    technologyAndReport.setReportId(report.getId());
                    technologyAndReport.setRuleId(rule.getId());
                    technologyAndReport.setId(UUIDUtil.getUUID());
                    technologyAndReports.add(technologyAndReport);
                }
            }else{
                depTree.setViolation(2);
            }

        }
        return depTree;
    }

    public class Counter {
        private int i = 0;

        void plus() {
            i++;
        }

        int getNumber() {
            return i;
        }
    }
}
