package com.zjty.inspect.service.impl;

import com.zjty.inspect.dao.TechnologyDao;
import com.zjty.inspect.dao.RuleDao;
import com.zjty.inspect.entity.Technology;
import com.zjty.inspect.entity.Rule;
import com.zjty.inspect.entity.RuleQo;
import com.zjty.inspect.service.RuleService;
import com.zjty.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 规则库
 *
 * @author Mcj
 * @date 2020-02-26 15:05
 */

@Service
@Transactional(rollbackFor = Exception.class)
public class RuleServiceImpl implements RuleService {

    /**
     * 规则库dao
     */
    @Autowired
    private RuleDao ruleDao;

    /**
     * 新增规则
     *
     * @param ruleQo 规则封装类
     */
    @Override
    public void addRule(RuleQo ruleQo) {
        Rule rule1 = ruleDao.findByTarget(ruleQo.getTarget());
        if(rule1!=null){
            return;
        }
        // TODO: 2020-02-27 适配技术与扫描关键字关联
        Rule rule = new Rule();
        rule.setTarget(ruleQo.getTarget());
        rule.setSuffix(ruleQo.getSuffix());
        rule.setTechnologyId(ruleQo.getTechnologyId());
        rule.setId(UUIDUtil.getUUID());
        ruleDao.save(rule);
    }

    /**
     * 条件查询+分页
     *
     * @param whereMap
     * @param page
     * @param size
     * @return
     */
    @Override
    public Page<Rule> findSearch(Map whereMap, int page, int size) {
        Specification<Rule> specification = createSpecification(whereMap);
        PageRequest pageRequest = PageRequest.of(page - 1, size);
        return ruleDao.findAll(specification, pageRequest);
    }

    /**
     * 动态条件构建
     *
     * @param searchMap
     * @return
     */
    private Specification<Rule> createSpecification(Map searchMap) {

        return new Specification<Rule>() {

            @Override
            public Predicate toPredicate(Root<Rule> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<Predicate>();
                if (searchMap.get("target") != null && !"".equals(searchMap.get("target"))) {
                    predicateList.add(cb.like(root.get("target").as(String.class), "%" + (String) searchMap.get("target") + "%"));
                }
                return cb.and(predicateList.toArray(new Predicate[predicateList.size()]));

            }
        };

    }

    @Override
    public void upRule(RuleQo ruleQo) {
        Rule rule = new Rule();
        rule.setTarget(ruleQo.getTarget());
        rule.setSuffix(ruleQo.getSuffix());
        rule.setTechnologyId(ruleQo.getTechnologyId());
        rule.setId(ruleQo.getId());
        ruleDao.save(rule);
    }

    @Override
    public void deleteById(String id) {
        ruleDao.deleteById(id);
    }

    @Override
    @Modifying
    public void deleteRule(RuleQo ruleQo) {
        ruleDao.deleteById(ruleQo.getId());
    }

    /**
     * 查询所有规则
     *
     * @return RuleVoList
     */
    @Override
    public List<Rule> findAll() {
        return ruleDao.findAll();
    }

    @Override
    public List<Rule> findByName(String name) {
        List<Rule> rules = ruleDao.findAllByTargetLike("%" + name + "%");
        if (rules != null && rules.size() > 0) {
            return rules;
        }
        return null;
    }
}
