package com.zjty.inspect.service.impl;

import com.zjty.inspect.dao.RuleDao;
import com.zjty.inspect.dao.TechnologyDao;
import com.zjty.inspect.entity.Rule;
import com.zjty.inspect.entity.TechnologyQo;
import com.zjty.inspect.entity.Technology;
import com.zjty.inspect.service.TechnologyService;
import com.zjty.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 适配技术
 *
 * @author Mcj
 * @date 2020-02-27 12:00
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class TechnologyServiceImpl implements TechnologyService {

    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyDao technologyDao;

    /**
     * 新增适配技术
     *
     * @param technologyQo 前端数据封装类
     */
    @Override
    public void addAdvice(TechnologyQo technologyQo) {
        Technology technology = new Technology();
        technology.setFund(technologyQo.getFund());
        technology.setAdvice(technologyQo.getAdvice());
        technology.setTechnologyName(technologyQo.getTechnologyName());
        if (technologyQo.getId() == null) {
            technology.setId(UUIDUtil.getUUID());
            Technology te = technologyDao.findAllByTechnologyNameEquals(technologyQo.getTechnologyName());
            if(te!=null){
                return;
            }
        } else {
            technology.setId(technologyQo.getId());
        }
        technologyDao.save(technology);
    }

    @Override
    public void add(Technology technology) {
        if (technology.getId() == null) {
            technology.setId(UUIDUtil.getUUID());
            Technology te = technologyDao.findAllByTechnologyNameEquals(technology.getTechnologyName());
            if(te!=null){
                return;
            }
        } else {
            technology.setId(technology.getId());
        }
        technologyDao.save(technology);
    }

    @Override
    public List<Technology> findAllTechnology() {
        return technologyDao.findAll();
    }

    @Override
    public List<String> findAllTechnologyNames() {
        return technologyDao.getTechnologyNames();
    }

    @Override
    public List<Technology> findAllTechnology(String name) {
        List<Technology> allByTechnologyNameLike = technologyDao.findAllByTechnologyNameLike("%" + name + "%");
        if(allByTechnologyNameLike==null || allByTechnologyNameLike.isEmpty()){
            return new ArrayList<>();
        }
        return technologyDao.findAllByTechnologyNameLike("%"+name+"%");
    }

    /**
     * 更新技术细节
     * @param technology
     */
    @Override
    public void update(Technology technology) {
        List<Rule> rules = ruleDao.findAllByTechnologyId(technology.getId());
        for (Rule rule : rules) {
            rule.setTechnologyName(technology.getTechnologyName());
        }
        ruleDao.saveAll(rules);

        technologyDao.save(technology);
    }

    @Override
    public Page<Technology> findSearch(Map whereMap, int page, int size) {
        Specification<Technology> specification = createSpecification(whereMap);
        PageRequest pageRequest =  PageRequest.of(page-1, size);
        return technologyDao.findAll(specification, pageRequest);
    }

    @Override
    public int findAllTechnologyCount() {
        return (int) technologyDao.count();
    }

    @Override
    public int findAllTechnologyNotSupport() {
        return technologyDao.countAllBySupportIsNot(1);
    }

    @Override
    public Technology findByid(String technologyId) {
        Optional<Technology> optional = technologyDao.findById(technologyId);
        if(optional.isPresent()){
            return optional.get();
        }
        return null;
    }

    /**
            * 动态条件构建
	 * @param searchMap
	 * @return
             */
    private Specification<Technology> createSpecification(Map searchMap) {

        return new Specification<Technology>() {

            @Override
            public Predicate toPredicate(Root<Technology> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<Predicate>();
                if (searchMap.get("technologyName")!=null && !"".equals(searchMap.get("technologyName"))) {
                    predicateList.add(cb.like(root.get("technologyName").as(String.class), "%"+(String)searchMap.get("technologyName")+"%"));
                }
                return cb.and( predicateList.toArray(new Predicate[predicateList.size()]));

            }
        };

    }
}
