package com.zjty.inspect.service.impl;

import com.zjty.inspect.dao.TechnologyDao;
import com.zjty.inspect.dao.RuleDao;
import com.zjty.inspect.entity.ExcelDataVo;
import com.zjty.inspect.entity.Technology;
import com.zjty.inspect.entity.Rule;
import com.zjty.inspect.entity.RuleQo;
import com.zjty.inspect.service.RuleService;
import com.zjty.inspect.service.TechnologyService;
import com.zjty.inspect.utils.ExcelReader;
import com.zjty.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 规则库
 *
 * @author Mcj
 * @date 2020-02-26 15:05
 */

@Service
@Transactional(rollbackFor = Exception.class)
public class RuleServiceImpl implements RuleService {

    /**
     * 规则库dao
     */
    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyService technologyService;

    @Override
    public void test() {
        List<ExcelDataVo> excelDataVos = ExcelReader.readExcel("C:\\Users\\Jey\\Desktop\\生成代码\\规则表.xlsx");
        System.out.println(excelDataVos);
        HashSet<String> set=new HashSet();
        for (ExcelDataVo excelDataVo : excelDataVos) {
            List<Rule> rules = ruleDao.findAllByTarget(excelDataVo.getName());
            if(rules==null||rules.size()==0){
                set.add(excelDataVo.getName());
            }
        }
        for (String s : set) {
            Rule rule=new Rule();
            rule.setTechnologyName("国产化依赖(支持)");
            rule.setTechnologyId("967a8bd0-eb27-4fb6-ba1d-1bca21cea5b4");
            rule.setSuffix("java");
            rule.setTarget(s);
            rule.setId(UUIDUtil.getUUID());
            ruleDao.save(rule);
        }

    }

    /**
     * 新增规则
     *
     * @param ruleQo 规则封装类
     */
    @Override
    public void addRule(RuleQo ruleQo) {
        List<String> suffixes = ruleQo.getSuffix();
        for (String suffix : suffixes) {
            Rule rule1 = ruleDao.findByTargetAndSuffixEqualsAndTechnologyIdEquals(ruleQo.getTarget(), suffix,ruleQo.getTechnologyId());
            if (rule1 != null) {
                return;
            }
            Rule rule = new Rule();
            rule.setTarget(ruleQo.getTarget());
            rule.setSuffix(suffix);
            rule.setTechnologyId(ruleQo.getTechnologyId());
            rule.setTechnologyName(ruleQo.getTechnologyName());
            rule.setId(UUIDUtil.getUUID());
            ruleDao.save(rule);
        }

    }

    /**
     * 条件查询+分页
     *
     * @param whereMap
     * @param page
     * @param size
     * @return
     */
    @Override
    public Page<Rule> findSearch(Map whereMap, int page, int size) {
        Specification<Rule> specification = createSpecification(whereMap);
        PageRequest pageRequest = PageRequest.of(page - 1, size);
        return ruleDao.findAll(specification, pageRequest);
    }

    /**
     * 动态条件构建
     *
     * @param searchMap
     * @return
     */
    private Specification<Rule> createSpecification(Map searchMap) {

        return new Specification<Rule>() {

            @Override
            public Predicate toPredicate(Root<Rule> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<Predicate>();
                if (searchMap.get("technologyName") != null && !"".equals(searchMap.get("technologyName"))) {
                    predicateList.add(cb.like(root.get("technologyName").as(String.class), "%" + (String) searchMap.get("technologyName") + "%"));
                }
                return cb.and(predicateList.toArray(new Predicate[predicateList.size()]));

            }
        };

    }

    @Override
    public void upRule(RuleQo ruleQo) {
        List<Rule> rules = ruleDao.findAllByTechnologyIdEqualsAndTargetEquals(ruleQo.getTechnologyId(), ruleQo.getTarget());
        for (Rule rule : rules) {
            ruleDao.deleteById(rule.getId());
        }
        addRule(ruleQo);
    }

    @Override
    public void deleteById(String id) {
        ruleDao.deleteById(id);
    }

    @Override
    @Modifying
    public void deleteRule(RuleQo ruleQo) {
        ruleDao.deleteById(ruleQo.getId());
    }

    /**
     * 查询所有规则
     *
     * @return RuleVoList
     */
    @Override
    public List<Rule> findAll() {
        return ruleDao.findAll();
    }

    @Override
    public List<Rule> findByName(String name) {
        List<Rule> rules = ruleDao.findAllByTargetLike("%" + name + "%");
        if (rules != null && rules.size() > 0) {
            return rules;
        }
        return null;
    }
}
