package com.zjty.inspect.inspect;

import com.zjty.inspect.dao.*;
import com.zjty.inspect.entity.*;
import com.zjty.inspect.enums.DependenceManagement;
import com.zjty.inspect.enums.Framework;
import com.zjty.inspect.enums.Language;
import com.zjty.inspect.enums.RecastMethod;
import com.zjty.inspect.utils.*;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告VO
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
@Service("property")
@Slf4j
@Transactional()
public class Inspector {

    @Autowired
    private AnalysisFile analysisFile;
    @Autowired
    private MavenUtil mavenUtil;
    @Autowired
    private BudgetUitl budgetUitl;

    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyDao technologyDao;
    @Autowired
    private ParameterDao parameterDao;

    private DependencyVo dependencyVo = new DependencyVo();
    private ArrayList<Rule> rules = new ArrayList<Rule>();
    private ArrayList<Warn> warns = new ArrayList<>();

    /**
     * 添加规则时去重使用
     * key：mysql-connect  :  *
     * mysql-connect:*
     * mysql-connect:.java
     * <p>
     * value:随意
     */
    private HashMap<String, Rule> ruleMap = new HashMap<>();

    /**
     * 支持的国产化技术
     */
    private Technology techJavaSupport;
    /**
     * 不支持的国产化技术
     */
    private Technology techNotCnSupport;

    /**
     * 未知依赖
     */
    private Technology techUnKnowSupport;
    /**
     * 计算预算的参数对象
     */
    private InspectParameter inspectParameter;

    private double codeSize = 0;
    /**
     * 报告对象
     */
    private ReportVo report;

    /**
     * 统计语言
     * key：java
     * value：10
     */
    private Map<String, Counter> languageMatchMap = new HashMap<>();

    /**
     * 后缀语言
     * key：properties
     * value:[{/Users/path},{/Users/path}]
     */
    private Map<String, Language> suffixLanguageMapping = new HashMap<>();

    /**
     * 规则列表
     */
    private List<Rule> ruleList;

    /**
     * 配置文件后缀
     * xml：list【路径】
     */

    private Map<String, List<Path>> configFileTypePathsMapping = new HashMap<>();

    private Map<String, List<Path>> ruleSuffixFileMap;
    private Map<String, List<Rule>> ruleSuffixMap;

    /**
     * 规则关联的技术
     * key:技术id
     * value:Technology
     */
    private Map<String, Technology> technologyHashMap = new HashMap<>();

    /**
     * 统计各后缀文件路径与出现次数,顺便解析jar与js文件
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     * 准备工作
     * 1.解析文件
     * 2/记录文件地址
     * 3/统计各个规则文件后缀
     * @return
     */
    public ReportVo inspect() {
        ruleSuffixFileMap = new HashMap<>();
        ruleSuffixMap = new HashMap<>();
        warns.clear();
        rules.clear();
        //查询技术，构造支持与非支持技术对象
        findExistTechnology();
        statisticsLanguage();
        this.ruleList = ruleDao.findAll();
        //统计项目组成文件构成
        //统计配置文件地址
        statisticsConfigFile();
        try {
            //以下为计算文件名称匹配正则表达式
            FileSystem aDefault = FileSystems.getDefault();
            Map<String, PathMatcher> languageSuffixMatcherMapping = new HashMap<>(16);
            //构造各个语言后缀文件的正则表达式
            for (String s : suffixLanguageMapping.keySet()) {
                languageSuffixMatcherMapping.put(s, aDefault.getPathMatcher("glob:**/*." + s));
            }
            //构造各个配置文件的正则表达式，用于解析依赖
            Map<PathMatcher, String> configFileMatcherSuffixMapping = new HashMap<>(16);
            for (String s : configFileTypePathsMapping.keySet()) {
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            //构造规则后缀的正则表达式
            Map<PathMatcher, String> ruleSuffix = new HashMap<>(16);
            for (String s : ruleSuffixFileMap.keySet()) {
                ruleSuffix.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            //文件读取
            Files.walkFileTree(Paths.get(inspectParameter.getPath()), new FileVisitor<Path>() {
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    //这里是对于路径（文件夹）的过滤，在这里读不到文件如果能判断，可以返回FileVisitResult.SKIP_SUBTREE 不访问子目录
                    //if(dir.getFileName().startsWith("."))return FileVisitResult.SKIP_SUBTREE;
                    //if (dir.endsWith(".git"))return FileVisitResult.SKIP_SUBTREE;
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {

                    for (Map.Entry<String, PathMatcher> entry : languageSuffixMatcherMapping.entrySet()) {
                        //通过正则表达式匹配.java类型后缀文件，并+1
                        if (entry.getValue().matches(file)) {
                            long length = file.toFile().length();
                            codeSize += length / 1024;
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : configFileMatcherSuffixMapping.entrySet()) {
                        //通过配置文件正则表达式匹配.xml文件，记录文件地址
                        if (entry.getKey().matches(file)) {
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : ruleSuffix.entrySet()) {
                        //通过规则匹配后缀正则表达式匹配,记录匹配上的文件地址
                        if (entry.getKey().matches(file)) {
                            ruleSuffixFileMap.get(entry.getValue()).add(file);
                        }
                    }
                    //检查到普通jar包
                    if (file.toString().endsWith(".jar")) {
                        //新建一个pom对象
                        ProjectPom projectPom = new ProjectPom();
                        //截取jar名称

                        String patten = RegexUtil.patten(file.getFileName().toString());
                        //新建一个依赖对象
                        PomDependency pomDependency = new PomDependency();
                        pomDependency.setArtifactId(patten);
                        projectPom.getDependencies().add(pomDependency);
                        // TODO: 2020-03-04 界定rule唯一，修改数据，一条数据绑定两种技术
                        //当参数为1时代表上传者管理员，代码可绝对信任，将jar名称当作可支持依赖添加进规则库中
                        if (inspectParameter.getAdmin() == 1) {
                            //新建规则对象
                            Rule rule = new Rule();
                            //设置适配技术id
                            rule.setTechnologyId(techJavaSupport.getId());
                            rule.setTarget(patten);
                            //设置文件后缀
                            rule.setSuffix("*");
                            rule.setId(UUIDUtil.getUUID());
                            rule.setTechnologyName(techJavaSupport.getTechnologyName());
                            //做规则查询，不用去数据库查询

                            if (!ruleMap.containsKey(patten + ":" + rule.getSuffix())) {
                                rules.add(rule);
                                ruleMap.put(patten + ":" + rule.getSuffix(), rule);
                            }
                            //设置当前依赖为可支持
                            pomDependency.setSupport(1);
                        } else {
                            //为普通用户上传，依赖需要检查是否支持。
                            int i = valiWarn(ruleList, file, patten, 0);
                            //如果值为0则代表是有不支持技术到匹配
                            pomDependency.setSupport(i);
                        }
                        dependencyVo.add(projectPom);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        inspectParameter.setCodeSize((int) codeSize);
        log.info("inspect:源代码扫描完成,统计各个文件后缀完成");
        return analysis();
    }

    /**
     * 解析数据
     *
     * @return Report
     */
    @Transactional
    public ReportVo analysis() {
        DependencyVo dependencyVo = new DependencyVo();
        setReportLanguageAndFrame();

        //查询所有规则
        ruleTransform(report.getRecastMethod());

        //解析配置文件集合
        for (Map.Entry<String, List<Path>> entry : configFileTypePathsMapping.entrySet()) {
            switch (entry.getKey()) {
                /**
                 * 配置文件的一个类型，xml文件
                 */
                case "xml":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("pom.xml")) {
                            // TODO: 2020-02-28 解析maven树文件，设置依赖保存到redis
                            report.setManager(DependenceManagement.MAVEN.getStatus());
                            ProjectPom projectPom = analysisFile.analysisPom(path);

                            StringBuilder stringBuilder = new StringBuilder();
                            for (PomDependency dependency : projectPom.getDependencies()) {
                                setRule(path, stringBuilder, dependency);
                            }
                            dependencyVo.add(projectPom);
                        }
                    }
                    break;
                case "gradle":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("build.gradle")) {
                            ProjectPom projectPom = new ProjectPom();
                            report.setManager(DependenceManagement.GRADLE.getStatus());
                            List<PomDependency> pomDependencies = AnalysisFile.analysisGradle(path);
                            projectPom.setDependencies(pomDependencies);
                            //设置依赖
                            StringBuilder stringBuilder = new StringBuilder();
                            for (PomDependency dependency : pomDependencies) {
                                setRule(path, stringBuilder, dependency);
                            }
                            dependencyVo.add(projectPom);
                        }
                    }
                    break;
                default:
            }
        }
        //指定后缀到文件匹配关键字
        for (Map.Entry<String, List<Path>> entry : ruleSuffixFileMap.entrySet()) {
            //entry，key为后缀，value为文件列表
            String key = entry.getKey();
            //从ruleSuffixList获取指定后缀的规则列表
            List<Rule> rules = ruleSuffixMap.get(key);
            for (Path path1 : entry.getValue()) {
                try {
                    //如果文件类型为jar活着class则不读取
                    if (path1.toAbsolutePath().toString().endsWith("jar") || path1.toAbsolutePath().toString().endsWith("class")) {
                        continue;
                    }
                    valiWarn(rules, path1, path1.getFileName().toString(), 0);
                    //将文件的每一行都与规则匹配
                    List<String> strings = Files.readAllLines(path1);
                    for (int i = 0; i < strings.size(); i++) {
                        valiWarn(rules, path1, strings.get(i), i + 1);
                    }
                } catch (IOException e) {
                    log.error("解析{}出错,异常信息为{}", path1.toAbsolutePath().toString(), e.getMessage());
                }
            }
        }
        //将得到的告警信息根据技术id进行转换
        Set<String> collect = warns.stream().map(Warn::getTechnologyId).collect(Collectors.toSet());
        List<Technology> allById = technologyDao.findAllById(collect);
        //计算技术金额
        Integer fund = 0;
        for (Technology tech : allById) {
            fund += tech.getFund();
        }
        //计算预算
        if (inspectParameter.getValid() != null) {
            BudgetVo budget = budgetUitl.getBudget(fund, report, inspectParameter);
            report.setBudgets(budget);
        }
        parameterDao.save(inspectParameter);
        //填充地址(如果有)
        report.setGitAddress(inspectParameter.getGitAddress());
        //填充适配技术
        report.setTechnologies(allById);
        //填充依赖
        report.setDependencyVo(dependencyVo);
        //数据转换
        HashMap<String, Technology> map = new HashMap<>();
        for (Technology technology1 : allById) {
            map.put(technology1.getId(), technology1);
        }
        HashMap<String, List<Warn>> warnMap = getWarnMap(map);
        ruleDao.saveAll(rules);
        report.setWarnDetails(warnMap);
        return report;
    }


    private void setReportLanguageAndFrame() {
        String most = null;
        int mostStatus = 0;
        int maxnum = 0;
        //统计文件最多的
        for (Map.Entry<String, Counter> entry : languageMatchMap.entrySet()) {
            if (entry.getValue().getNumber() > maxnum) {
                most = suffixLanguageMapping.get(entry.getKey()).name();
                Language language = Language.valueOf(most);
                mostStatus = language.getStatus();
                maxnum = entry.getValue().getNumber();
            }
        }
        //设置语言
        report.setLanguage(most == null ? Language.UNKNOW.getStatus() : mostStatus);
        //设置架构
        report.setFramework(languageMatchMap.get("jsp").i > 0 ? Framework.混合型架构.getStatus() : Framework.分离型架构.getStatus());
        //设置是否需要重构
        if (languageMatchMap.get("jsp").i == 0 & languageMatchMap.get("java").i == 0) {
            report.setRecastMethod(RecastMethod.适配重构.getStatus());
            log.info("inspect:代码解析完成,建议进行");
        } else {
            report.setRecastMethod(RecastMethod.代码修改.getStatus());
        }
    }

    /**
     * rule所需要数据装配
     */
    private void ruleTransform(int i) {
        if (i == 1) {
            List<Technology> front = technologyDao.findAllByBackorfrontEquals("1");
            List<String> ids = front.stream().map(Technology::getId).collect(Collectors.toList());
            this.ruleList = ruleDao.findAllByTechnologyIdIn(ids);
        } else {
            if (inspectParameter.getRecastMethod() == 1) {
                this.ruleList = ruleDao.findAll();
            } else {
                List<Technology> front = technologyDao.findAllByBackorfrontEquals("1");
                List<String> ids = front.stream().map(Technology::getId).collect(Collectors.toList());
                this.ruleList = ruleDao.findAllByTechnologyIdIn(ids);
            }
        }
        Set<String> id = ruleList.stream().map(Rule::getTechnologyId).collect(Collectors.toSet());
        List<Technology> technologies = technologyDao.findAllByIdIn(new ArrayList<>(id));
        for (Technology technology : technologies) {
            technologyHashMap.put(technology.getId(), technology);
        }
        //根据后缀名，收集文件进行操作
        for (Rule rule : ruleList) {
            if (!ruleSuffixFileMap.containsKey(rule.getSuffix())) {
                ruleSuffixFileMap.put(rule.getSuffix(), new ArrayList<>());
            }
        }
        //根据后缀名进行规则收集
        for (Rule rule : ruleList) {
            if (!ruleSuffixMap.containsKey(rule.getSuffix())) {
                ruleSuffixMap.put(rule.getSuffix(), new ArrayList<>());
                ruleSuffixMap.get(rule.getSuffix()).add(rule);
            } else {
                ruleSuffixMap.get(rule.getSuffix()).add(rule);
            }
        }
    }

    private void statisticsConfigFile() {
        this.configFileTypePathsMapping = new HashMap<>();
        configFileTypePathsMapping.put("xml", new ArrayList<>());
        configFileTypePathsMapping.put("json", new ArrayList<>());
        configFileTypePathsMapping.put("gradle", new ArrayList<>());
        configFileTypePathsMapping.put("properties", new ArrayList<>());
        configFileTypePathsMapping.put("yml", new ArrayList<>());
    }

    private void statisticsLanguage() {
        for (String languageName : suffixLanguageMapping.keySet()) {
            //配置如：java，0
            languageMatchMap.put(languageName, new Counter());
        }
    }

    private void findExistTechnology() {
        techJavaSupport = technologyDao.findAllByTechnologyNameEquals("国产化依赖(支持)");
        techNotCnSupport = technologyDao.findAllByTechnologyNameEquals("非国产化依赖(不支持)");
        techUnKnowSupport = technologyDao.findAllByTechnologyNameEquals("未知依赖(未知)");
    }

    private HashMap<String, List<Warn>> getWarnMap(HashMap<String, Technology> map) {
        HashMap<String, List<Warn>> warnMap = new HashMap<>();
        for (Warn warn : warns) {
            if (!warnMap.containsKey(map.get(warn.getTechnologyId()).getTechnologyName())) {
                ArrayList<Warn> warns1 = new ArrayList<>();
                warns1.add(warn);
                warnMap.put(map.get(warn.getTechnologyId()).getTechnologyName(), warns1);
            } else {
                warnMap.get(map.get(warn.getTechnologyId()).getTechnologyName()).add(warn);
            }
        }
        return warnMap;
    }

    /**
     * @param path          文件路径
     * @param stringBuilder string缓冲区
     * @param dependency    依赖
     */
    private void setRule(Path path, StringBuilder stringBuilder, PomDependency dependency) {

        stringBuilder.append(dependency.getGroupId()).append(":").append(dependency.getArtifactId());
        if (inspectParameter.getAdmin() == 1) {
            Rule rule = new Rule();
            rule.setTechnologyId(techJavaSupport.getId());
            rule.setTarget(dependency.getGroupId());
            rule.setTechnologyName(techJavaSupport.getTechnologyName());
            rule.setSuffix("*");
            rule.setId(UUIDUtil.getUUID());
            Rule rule1 = new Rule();
            rule1.setTechnologyId(techJavaSupport.getId());
            rule1.setTarget(dependency.getArtifactId());
            rule1.setTechnologyName(techJavaSupport.getTechnologyName());
            rule1.setSuffix("*");
            rule1.setId(UUIDUtil.getUUID());
            if (!ruleMap.containsKey(dependency.getGroupId() + ":" + rule.getSuffix())) {
                rules.add(rule);
                ruleMap.put(dependency.getGroupId() + ":" + rule.getSuffix(), rule);
            }
            if (!ruleMap.containsKey(dependency.getArtifactId() + ":" + rule1.getSuffix())) {
                rules.add(rule1);
                ruleMap.put(dependency.getArtifactId() + ":" + rule1.getSuffix(), rule1);
            }
            dependency.setSupport(1);
        } else {
            int i = valiWarn(ruleList, path, stringBuilder.toString(), 0);
            dependency.setSupport(i);
        }
        stringBuilder.delete(0, stringBuilder.length());
    }


    /**
     * 将数据与规则进行匹配，得出当前依赖是否支持
     *
     * @param rules 规则
     * @param path  匹配的文件
     * @param data  行数据
     * @param num   行数
     * @return 1：正常、2：不支持依赖
     */
    public int valiWarn(List<Rule> rules, Path path, String data, int num) {
        //默认未知依赖
        int supportStatus = 3;
        for (Rule rule : rules) {
            //规则目标字符串为null 或者 空时
            if (rule.getTarget() == null || rule.getTarget().isEmpty()) {
                continue;
            }
            //匹配字符串
            int index = KmpUtil.kmpMatch(data, rule.getTarget());
            //当index>0时代表data中有当前规则
            if (index > -1) {
                //判断当前规则是否是不支持规则
                Warn warn = new Warn();
                //如果是不支持规则则需要保存匹配信息
                warn.setFilePath(path.toAbsolutePath().toString());
                warn.setLineNum(num);
                warn.setRuleId(rule.getId());
                warn.setRule(rule.getTarget());
                warn.setTechnologyId(rule.getTechnologyId());
                warn.setTechnologyName(technologyHashMap.get(rule.getTechnologyId()).getTechnologyName());
                if (technologyHashMap.get(rule.getTechnologyId()).getSupport() == 2) {
                    warns.add(warn);
                    //设置a=0代表当前依赖有问题
                    supportStatus = 2;
                } else if (technologyHashMap.get(rule.getTechnologyId()).getSupport() == 3) {
                    warns.add(warn);
                    supportStatus = 3;
                } else if (technologyHashMap.get(rule.getTechnologyId()).getSupport() == 1) {
                    warns.add(warn);
                    supportStatus = 1;
                }
            }
        }
        return supportStatus;
    }

    public class Counter {
        private int i = 0;

        void plus() {
            i++;
        }

        int getNumber() {
            return i;
        }
    }
}
