package com.zjty.inspect.utils;

import com.zjty.inspect.dao.*;
import com.zjty.inspect.entity.*;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.nio.charset.MalformedInputException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告VO
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
@Service
public class Inspector {

    @Autowired
    private AnalysisFile analysisFile;
    @Autowired
    private MavenUtil mavenUtil;
    @Autowired
    private BudgetUitl budgetUitl;

    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyDao technologyDao;
    @Autowired
    private ParameterDao parameterDao;

    private DepTreeVo depTreeVo = new DepTreeVo();
    private ArrayList<Rule> rules = new ArrayList<Rule>();
    private ArrayList<Warn> warns = new ArrayList<>();

    /**
     * 添加规则时去重使用
     * key：mysql-connect:*
     * value:随意
     */
    private HashMap<String,Rule> ruleMap = new HashMap<>();

    /**
     * 支持的国产化技术
     */
    Technology techJavaSupport;
    /**
     * 不支持的国产化技术
     */
    Technology techNotCnSupport;
    /**
     * 计算预算的参数对象
     */
    private InspectParameter inspectParameter;

    /**
     * 报告对象
     */
    private ReportVo report;

    /**
     * 统计语言
     * key：java
     * value：10
     */
    private Map<String, Counter> languageMatchMap = new HashMap<>();

    /**
     * 后缀语言
     * key：properties
     * value:[{/Users/path},{/Users/path}]
     */
    private Map<String, Report.Language> suffixLanguageMapping = new HashMap<>();

    /**
     * 规则列表
     */
    private List<Rule> ruleList;

    /**
     * 配置文件后缀
     * xml：list【路径】
     */
    private Map<String, List<Path>> configFileTypePathsMapping = new HashMap<>();

    private Map<String, List<Path>> ruleSuffixMap = new HashMap<>();
    private Map<String, List<Rule>> ruleSuffixList = new HashMap<>();

    /**
     * 统计各后缀文件路径与出现次数,顺便解析jar与js文件
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     *
     * @return 报告
     */
    public ReportVo inspect() {
        //查询技术，构造支持与非支持技术对象
        techJavaSupport = technologyDao.findAllByTechnologyNameEquals("Java依赖");
        techNotCnSupport = technologyDao.findAllByTechnologyNameEquals("非国产化依赖");
        //统计项目组成文件构成
        for (String languageName : suffixLanguageMapping.keySet()) {
            //配置如：java，0
            languageMatchMap.put(languageName, new Counter());
        }
        //统计文件地址
        this.configFileTypePathsMapping = new HashMap<>();
        configFileTypePathsMapping.put("xml", new ArrayList<>());
        configFileTypePathsMapping.put("json", new ArrayList<>());
        configFileTypePathsMapping.put("gradle", new ArrayList<>());
        configFileTypePathsMapping.put("properties", new ArrayList<>());
        configFileTypePathsMapping.put("yml", new ArrayList<>());

        //查询所有规则
        this.ruleList = ruleDao.findAll();

        for (Rule rule : ruleList) {

            if (!ruleSuffixMap.containsKey(rule.getSuffix())) {
                //用于存储各种后缀的文件集合
                ruleSuffixMap.put(rule.getSuffix(), new ArrayList<>());

                //将每个后缀的规则分类
                //key:规则后缀
                //value:规则集合
                ruleSuffixList.put(rule.getSuffix(), new ArrayList<>());
                ruleSuffixList.get(rule.getSuffix()).add(rule);
            } else {
                ruleSuffixList.get(rule.getSuffix()).add(rule);
            }
            //规则添加时去重使用，mysql-connect:*
            ruleMap.put(rule.getTarget()+":"+rule.getSuffix(),rule);
        }

        try {
            //以下为计算文件名称匹配正则表达式
            FileSystem aDefault = FileSystems.getDefault();
            Map<String, PathMatcher> languageSuffixMatcherMapping = new HashMap<>(16);
            //构造各个语言后缀文件的正则表达式
            for (String s : suffixLanguageMapping.keySet()) {
                languageSuffixMatcherMapping.put(s, aDefault.getPathMatcher("glob:**/*." + s));
            }
            Map<PathMatcher, String> configFileMatcherSuffixMapping = new HashMap<>(16);
            for (String s : configFileTypePathsMapping.keySet()) {
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            Map<PathMatcher, String> ruleSuffix = new HashMap<>(16);
            for (String s : ruleSuffixMap.keySet()) {
                ruleSuffix.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            //文件读取
            Files.walkFileTree(Paths.get(inspectParameter.getPath()), new FileVisitor<Path>() {
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    //这里是对于路径（文件夹）的过滤，在这里读不到文件如果能判断，可以返回FileVisitResult.SKIP_SUBTREE 不访问子目录
                    //if(dir.getFileName().startsWith("."))return FileVisitResult.SKIP_SUBTREE;
                    //if (dir.endsWith(".git"))return FileVisitResult.SKIP_SUBTREE;
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {

                    for (Map.Entry<String, PathMatcher> entry : languageSuffixMatcherMapping.entrySet()) {
                        if (entry.getValue().matches(file)) {
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : configFileMatcherSuffixMapping.entrySet()) {
                        if (entry.getKey().matches(file)) {
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : ruleSuffix.entrySet()) {
                        if (entry.getKey().matches(file)) {
                            ruleSuffixMap.get(entry.getValue()).add(file);
                        }
                    }
                    //获取普通依赖jar
                    if (file.toString().endsWith(".jar")) {
                        ProjectPom projectPom = new ProjectPom();
                        // TODO: 2020-02-28 jar文件需要修改格式变为string
                        String[] split = file.getFileName().toString().split("\\.");
                        PomDependency pomDependency = new PomDependency();
                        pomDependency.setArtifactId(split[0]);
                        projectPom.getDependencies().add(pomDependency);
                        if(inspectParameter.getAdmin()==1){
                            Rule rule = new Rule();
                            rule.setTechnologyId(techJavaSupport.getId());
                            rule.setTarget(split[0]);
                            rule.setSuffix("*");
                            rule.setId(UUIDUtil.getUUID());
                            if(!ruleMap.containsKey(split[0]+":"+rule.getSuffix())){
                                rules.add(rule);
                                ruleMap.put(split[0]+":"+rule.getSuffix(),rule);
                            }
                            pomDependency.setSupport(1);
                        }else{
                            int i = valiWarn(ruleList, file, split[0], 0);
                            if(i==0){
                                pomDependency.setSupport(2);
                                projectPom.getDependencies().add(pomDependency);
                            }
                        }
                        depTreeVo.add(projectPom);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        }
        return analysis();
    }

    /**
     * 构成报告所需要到数据
     *
     * @return Report
     */
    @Transactional
    public ReportVo analysis() {
        DepTreeVo depTreeVo = new DepTreeVo();
        String most = null;
        int maxnum = 0;
        for (Map.Entry<String, Counter> entry : languageMatchMap.entrySet()) {
            if (entry.getValue().getNumber() > maxnum) {
                most = suffixLanguageMapping.get(entry.getKey()).name();
                maxnum = entry.getValue().getNumber();
            }
        }
        report.setLanguage(most == null ? Report.Language.UNKNOW.name() : most);
        report.setFramework(languageMatchMap.get("jsp").i > 0 ? "混合型架构" : "分离型架构");
        if (languageMatchMap.get("jsp").i == 0 & languageMatchMap.get("java").i == 0) {
            report.setRecastMethod(2);
        } else {
            report.setRecastMethod(1);
        }
        // TODO: 2020-02-28 jsFiles, jarFiles

        for (Map.Entry<String, List<Path>> entry : configFileTypePathsMapping.entrySet()) {
            switch (entry.getKey()) {
                /**
                 * 配置文件的一个类型，xml文件
                 */
                case "xml":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("pom.xml")) {
                            // TODO: 2020-02-28 解析maven树文件，设置依赖保存到redis
                            report.setManager(Report.DependenceManagement.MAVEN.name());
                            ProjectPom projectPom = analysisFile.analysisPom(path);

                            StringBuilder stringBuilder = new StringBuilder();
                            for (PomDependency dependency : projectPom.getDependencies()) {
                                stringBuilder.append(dependency.getGroupId()).append(":").append(dependency.getArtifactId());
                                if(inspectParameter.getAdmin()==1){
                                    Rule rule = new Rule();
                                    rule.setTechnologyId(techJavaSupport.getId());
                                    rule.setTarget(dependency.getGroupId());
                                    rule.setSuffix("*");
                                    rule.setId(UUIDUtil.getUUID());

                                    Rule rule1 = new Rule();
                                    rule1.setTechnologyId(techJavaSupport.getId());
                                    rule1.setTarget(dependency.getArtifactId());
                                    rule1.setSuffix("*");
                                    rule1.setId(UUIDUtil.getUUID());

                                    if(!ruleMap.containsKey(dependency.getGroupId()+":"+rule.getSuffix())){
                                        rules.add(rule);
                                        ruleMap.put(dependency.getGroupId()+":"+rule.getSuffix(),rule);
                                    }
                                    if(!ruleMap.containsKey(dependency.getArtifactId()+":"+rule1.getSuffix())){
                                        rules.add(rule1);
                                        ruleMap.put(dependency.getArtifactId()+":"+rule1.getSuffix(),rule1);
                                    }
                                    dependency.setSupport(1);
                                }else{
                                    int i = valiWarn(ruleList, path, stringBuilder.toString(), 0);
                                    if(i==0){
                                        dependency.setSupport(2);
                                    }
                                }
                                stringBuilder.delete(0,stringBuilder.length());
                            }
                            depTreeVo.add(projectPom);
                        }
                    }
                    break;
                case "gradle":
                    for (Path path : entry.getValue()) {
                        if (path.getFileName().endsWith("build.gradle")) {
                            ProjectPom projectPom = new ProjectPom();
                            report.setManager(Report.DependenceManagement.GRADLE.name());
                            List<PomDependency> pomDependencies = AnalysisFile.analysisGradle(path);
                            projectPom.setDependencies(pomDependencies);
                            //设置依赖
                            StringBuilder stringBuilder = new StringBuilder();
                            for (PomDependency dependency : pomDependencies) {
                                stringBuilder.append(dependency.getGroupId()).append(":").append(dependency.getArtifactId());
                                if(inspectParameter.getAdmin()==1){
                                    Rule rule = new Rule();
                                    rule.setTechnologyId(techJavaSupport.getId());
                                    rule.setTarget(dependency.getGroupId());
                                    rule.setSuffix("*");
                                    rule.setId(UUIDUtil.getUUID());

                                    Rule rule1 = new Rule();
                                    rule1.setTechnologyId(techJavaSupport.getId());
                                    rule1.setTarget(dependency.getArtifactId());
                                    rule1.setSuffix("*");
                                    rule1.setId(UUIDUtil.getUUID());
                                    if(!ruleMap.containsKey(dependency.getGroupId()+":"+rule.getSuffix())){
                                        rules.add(rule);
                                        ruleMap.put(dependency.getGroupId()+":"+rule.getSuffix(),rule);
                                    }

                                    if(!ruleMap.containsKey(dependency.getArtifactId()+":"+rule1.getSuffix())){
                                        rules.add(rule1);
                                        ruleMap.put(dependency.getArtifactId()+":"+rule1.getSuffix(),rule1);
                                    }
                                    dependency.setSupport(1);
                                }else{
                                    int i = valiWarn(ruleList, path, stringBuilder.toString(), 0);
                                    if(i==0){
                                        dependency.setSupport(2);
                                    }

                                }
                                stringBuilder.delete(0,stringBuilder.length());
                            }
                            depTreeVo.add(projectPom);
                        }
                    }
                default:
            }
        }
        //指定后缀到文件匹配关键字
        for (Map.Entry<String, List<Path>> entry : ruleSuffixMap.entrySet()) {
            String key = entry.getKey();
            List<Rule> rules = ruleSuffixList.get(key);
            for (Path path1 : entry.getValue()) {
                try {
                    if(path1.toAbsolutePath().toString().endsWith("jar") || path1.toAbsolutePath().toString().endsWith("class")){
                        continue;
                    }
                    List<String> strings = Files.readAllLines(path1);
                    for (int i = 0; i < strings.size(); i++) {
                        valiWarn(rules, path1, strings.get(i), i + 1);
                    }
                } catch (MalformedInputException e){

                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        Set<String> collect = warns.stream().map(Warn::getTechnologyId).collect(Collectors.toSet());
        List<Technology> allById = technologyDao.findAllById(collect);
        Integer fund = 0;
        for (Technology tech : allById) {
            fund += tech.getFund();
        }
        //计算预算，三种预算
        if (inspectParameter.getValid() != null) {
            List<Budget> budget = budgetUitl.getBudget(fund, inspectParameter);
            report.setBudgets(budget);
            inspectParameter.setId(UUIDUtil.getUUID());
            parameterDao.save(inspectParameter);
        }
        report.setGitAddress(inspectParameter.getGitAddress());
        report.setTechnologies(allById);
        report.setDepTreeVo(depTreeVo);
        HashMap<String, Technology> map = new HashMap<>();
        for (Technology technology1 : allById) {
            map.put(technology1.getId(),technology1);
        }

        HashMap<String, List<Warn>> warnMap = new HashMap<>();
        for (Warn warn : warns) {
            if(!warnMap.containsKey(map.get(warn.getTechnologyId()).getTechnologyName())){
                ArrayList<Warn> warns1 = new ArrayList<>();
                warns1.add(warn);
                warnMap.put(map.get(warn.getTechnologyId()).getTechnologyName(),warns1);
            }else{
                warnMap.get(map.get(warn.getTechnologyId()).getTechnologyName()).add(warn);
            }
        }
        ruleDao.saveAll(rules);
        report.setWarnDetails(warnMap);
        // TODO: 2020-02-28 生成报告，返回地址 report.setsourceaddress;
        return report;
    }

    public int valiWarn(List<Rule> rules, Path path, String name, int num) {
        int a = 1;
        for (Rule rule : rules) {
            if(name.isEmpty()){
                continue;
            }
            int ik = KmpUtil.kmpMatch(name, rule.getTarget());
            if (ik > 0) {
                //rule匹配上了
                if(rule.getTechnologyId().equals(techNotCnSupport.getId())){
                    Warn warn = new Warn();
                    warn.setFilePath(path.toAbsolutePath().toString());
                    warn.setLineNum(num);
                    warn.setTechnologyId(rule.getTechnologyId());
                    warn.setRuleId(rule.getId());
                    warn.setRule(rule.getTarget());
                    warns.add(warn);
                    a=0;
                }
            }
        }
        return a;
    }
    public class Counter {
        private int i = 0;

        void plus() {
            i++;
        }

        int getNumber() {
            return i;
        }
    }
}
