package com.zjty.inspect.service.impl;

import com.zjty.inspect.dao.MasterRuleDao;
import com.zjty.inspect.dao.TechnologyDao;
import com.zjty.inspect.entity.MasterTechnology;
import com.zjty.inspect.entity.MasterRule;
import com.zjty.inspect.entity.TechnologyQo;
import com.zjty.inspect.service.TechnologyService;
import com.zjty.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 适配技术
 *
 * @author Mcj
 * @date 2020-02-27 12:00
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class TechnologyServiceImpl implements TechnologyService {

    @Autowired
    private MasterRuleDao ruleDao;
    @Autowired
    private TechnologyDao technologyDao;

    /**
     * 新增适配技术
     *
     * @param technologyQo 前端数据封装类
     */
    @Override
    public void addAdvice(TechnologyQo technologyQo) {
        MasterTechnology technology = new MasterTechnology();
        technology.setFund(technologyQo.getFund());
        technology.setAdvice(technologyQo.getAdvice());
        technology.setTechnologyName(technologyQo.getTechnologyName());
        if (technologyQo.getId() == null) {
            technology.setId(UUIDUtil.getUUID());
            MasterTechnology te = technologyDao.findAllByTechnologyNameEquals(technologyQo.getTechnologyName());
            if(te!=null){
                return;
            }
        } else {
            technology.setId(technologyQo.getId());
        }
        technologyDao.save(technology);
    }

    @Override
    public void add(MasterTechnology technology) {
        if (technology.getId() == null) {
            technology.setId(UUIDUtil.getUUID());
            MasterTechnology te = technologyDao.findAllByTechnologyNameEquals(technology.getTechnologyName());
            if(te!=null){
                return;
            }
        } else {
            technology.setId(technology.getId());
        }
        technologyDao.save(technology);
    }

    @Override
    public void addTech(MasterTechnology technology) {
        Optional<MasterTechnology> optional = technologyDao.findById(technology.getId());
        if(optional.isPresent()){
            MasterTechnology te = optional.get();
            if(te.getTechnologyName().equals(technology.getTechnologyName())){
                return;
            }
        }

    }

    @Override
    public List<MasterTechnology> findAllTechnology() {
        return technologyDao.findAll();
    }

    @Override
    public List<String> findAllTechnologyNames() {
        return technologyDao.getTechnologyNames();
    }

    @Override
    public List<MasterTechnology> findAllTechnology(String name) {
        List<MasterTechnology> allByTechnologyNameLike = technologyDao.findAllByTechnologyNameLike("%" + name + "%");
        if(allByTechnologyNameLike==null || allByTechnologyNameLike.isEmpty()){
            return new ArrayList<>();
        }
        return technologyDao.findAllByTechnologyNameLike("%"+name+"%");
    }

    /**
     * 更新技术细节
     * @param technology
     */
    @Override
    public void update(MasterTechnology technology) {
        List<MasterRule> rules = ruleDao.findAllByTechnologyId(technology.getId());
        for (MasterRule rule : rules) {
            rule.setTechnologyName(technology.getTechnologyName());
        }
        ruleDao.saveAll(rules);

        technologyDao.save(technology);
    }

    @Override
    public Page<MasterTechnology> findSearch(Map whereMap, int page, int size) {
        Specification<MasterTechnology> specification = createSpecification(whereMap);
        PageRequest pageRequest =  PageRequest.of(page-1, size);
        return technologyDao.findAll(specification, pageRequest);
    }

    @Override
    public int findAllTechnologyCount() {
        return (int) technologyDao.count();
    }

    @Override
    public int findAllTechnologyNotSupport() {
        return technologyDao.countAllBySupportIsNot(1);
    }

    @Override
    public MasterTechnology findByid(String technologyId) {
        Optional<MasterTechnology> optional = technologyDao.findById(technologyId);
        if(optional.isPresent()){
            return optional.get();
        }
        return null;
    }

    /**
            * 动态条件构建
	 * @param searchMap
	 * @return
             */
    private Specification<MasterTechnology> createSpecification(Map searchMap) {

        return new Specification<MasterTechnology>() {

            @Override
            public Predicate toPredicate(Root<MasterTechnology> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<Predicate>();
                if (searchMap.get("technologyName")!=null && !"".equals(searchMap.get("technologyName"))) {
                    predicateList.add(cb.like(root.get("technologyName").as(String.class), "%"+(String)searchMap.get("technologyName")+"%"));
                }
                return cb.and( predicateList.toArray(new Predicate[predicateList.size()]));

            }
        };

    }
}
