package com.zjty.inspect.controller;

import com.zjty.inspect.aop.AuthAnnotation;
import com.zjty.inspect.entity.*;
import com.zjty.inspect.service.RuleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Map;

/**
 * 规则库设置
 *
 * @author Mcj
 * @date 2020-02-26 14:42
 */
@Slf4j
@RestController
@RequestMapping("/rule")
@Api(value = "规则页面管理接口", description = "规则页面管理接口,提供页面的增、删、改、查")
public class RuleController {

    @Autowired
    private RuleService ruleService;

    /**
     * test
     * 新增规则
     *
     * @param
     * @return
     */
    @PostMapping
    @ApiOperation("新增规则")
    @AuthAnnotation(code = {"001200", "001400"})
    public ServerResponse<Rule> rule(@RequestBody Rule rule) {
        log.info("规则{}", rule);
        Rule newRule = ruleService.addRulePlus(rule);
        return ServerResponse.ok(newRule);
    }

    /**
     * 修改规则
     *
     * @param rule 规则封装
     * @return
     */
    @PostMapping(value = "/update")
    @ApiOperation("修改规则")
    @AuthAnnotation(code = {"001200", "001400"})
    public ServerResponse update(@RequestBody Rule rule) {
        ruleService.upRulePlus(rule);
        return ServerResponse.ok(200);
    }

    /**
     * 根据id删除规则
     *
     * @param id
     * @return
     */
    @ApiOperation("根据数据封装删除规则")
    @DeleteMapping(value = "/{id}")
    @AuthAnnotation(code = {"001200", "001400"})
    public ServerResponse deleteByQo(@PathVariable String id) {
        ruleService.deleteRulePlus(id);
        return ServerResponse.ok(200);
    }


    /**
     * 分页+多条件查询
     *
     * @param searchMap 查询条件封装
     * @param page      页码
     * @param size      页大小
     * @return 分页结果
     */
    @ApiOperation("分页查询页面列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "page", value = "页码", required = true, paramType = "path", dataType = "int"),
            @ApiImplicitParam(name = "size", value = "每页记录数", required = true, paramType = "path", dataType = "int")
    })
    @RequestMapping(value = "/search/{page}/{size}", method = RequestMethod.POST)
    @AuthAnnotation(code = {"001200", "001400"})
    public ServerResponse<PageResult<Rule>> findSearch(@RequestBody Map searchMap, @PathVariable int page, @PathVariable int size) {
        Page<Rule> pageList = ruleService.findSearch(searchMap, page, size);
        return ServerResponse.ok(new PageResult<>(pageList.getTotalElements(), pageList.getContent()));
    }

    /**
     * 导出所有规则
     *
     * @return 规则
     */
    @ApiOperation("导出所有规则")
    @GetMapping("/export")
    public ServerResponse exportAllRules() {
        String path = ruleService.exportData();
        return ServerResponse.ok(path);
    }

    @PostMapping("/importRules")
    @ApiOperation("导入所有规则")
    @AuthAnnotation(code = {"001200"})
    public ServerResponse importRules(@RequestParam("file") MultipartFile file) {
        ruleService.importRules(file);
        return ServerResponse.ok("");
    }
}
