package com.zjty.inspect.utils;

import com.zjty.inspect.entity.*;
import com.zjty.inspect.entity.Properties;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.transaction.annotation.Transactional;

import java.io.IOException;
import java.nio.charset.MalformedInputException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;

/**
 * 项目体检，根据既定特征值，
 * 扫描、统计、分析项目特征，
 * 生成报告VO
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
public class Inspector {


    /**
     * 报告对象
     */
    private Report report;

    /**
     * 语言
     */
    private Map<String, Counter> languageMatchMap ;

    /**
     * 后缀语言
     */
    private Map<String, Report.Language> suffixLanguageMapping;
    /**
     * 路径
     */
    private String path;

    /**
     * 规则列表
     */
    private List<Rule> ruleList;

    /**
     * 配置文件后缀
     * xml：list【路径】
     */
    private Map<String, List<Path>> configFileTypePathsMapping;

    /**
     * 警告，建议
     */
    private List<Warn> warnList;

    /**
     * 后端依赖
     */
    private List<PomDependency> jarFiles;

    /**
     * 前端依赖
     */
    private List<PomDependency> jsFiles;

    /**
     * 版本version
     */
    private HashMap<String,String> versionMap = new HashMap<>();

    public Inspector(String path, Map<String, Report.Language> suffixLanguageMapping, List<Rule> ruleList) {
        this.path = path;
        //内存中配置的语言map
        this.suffixLanguageMapping = suffixLanguageMapping;
        this.ruleList = ruleList;
        this.languageMatchMap = new HashMap<>();
        for (String s : suffixLanguageMapping.keySet()) {
            //配置如：java，0
            languageMatchMap.put(s, new Counter());
        }
        /**
         * 如果存在html文件，不一定是前端代码，
         * 如果只存在html文件，不存在其他类型文件，大概率是前端代码
         * 策略：
         *      先检测是否存在html文件，如果存在，假定为前端项目
         *      然后检测其他条件，如果符合，将假定冲掉
         */
        this.configFileTypePathsMapping = new HashMap<>();
        configFileTypePathsMapping.put("xml", new ArrayList<>());
        configFileTypePathsMapping.put("gradle", new ArrayList<>());
        configFileTypePathsMapping.put("properties", new ArrayList<>());
        configFileTypePathsMapping.put("yml", new ArrayList<>());
        this.jarFiles = new ArrayList<>();
        this.jsFiles = new ArrayList<>();
        this.report = new Report();
        this.warnList = new ArrayList<>();
    }

    int fileNum = 0;
    long lineNum = 0;

    /**
     * FileVisitResult.CONTINUE 继续遍历
     * FileVisitResult.TERMINATE 中止访问
     * FileVisitResult.SKIP_SIBLINGS 不访问同级的文件或目录
     * FileVisitResult.SKIP_SUBTREE 不访问子目录
     */
    public Report inspect() {
        try {
            FileSystem aDefault = FileSystems.getDefault();
            Map<String, PathMatcher> languageSuffixMatcherMapping = new HashMap<>();
            for (String s : suffixLanguageMapping.keySet()) {
                languageSuffixMatcherMapping.put(s, aDefault.getPathMatcher("glob:**/*." + s));
            }
            Map<PathMatcher, String> configFileMatcherSuffixMapping = new HashMap<>();
            for (String s : configFileTypePathsMapping.keySet()) {
                configFileMatcherSuffixMapping.put(aDefault.getPathMatcher("glob:**/*." + s), s);
            }
            /// TODO: 2020-02-12
            // WebSocketServer.sendIn(uuid,TimeUtil.getNowDate() + " 项目评估：" + "开始读取项目文件");
            Files.walkFileTree(Paths.get(path), new FileVisitor<Path>() {
                /**
                 * 统计某个后缀出现的次数
                 * @param dir
                 * @param attrs
                 * @return
                 * @throws IOException
                 */
                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    /**
                     * 这里是对于路径（文件夹）的过滤，在这里读不到文件
                     * 如果能判断，可以返回FileVisitResult.SKIP_SUBTREE 不访问子目录
                     */
                    //if(dir.getFileName().startsWith("."))return FileVisitResult.SKIP_SUBTREE;
                    //if (dir.endsWith(".git"))return FileVisitResult.SKIP_SUBTREE;
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    fileNum++;
                    for (Map.Entry<String, PathMatcher> entry : languageSuffixMatcherMapping.entrySet()) {
                        if (entry.getValue().matches(file)) {
                            languageMatchMap.get(entry.getKey()).plus();
                        }
                    }
                    for (Map.Entry<PathMatcher, String> entry : configFileMatcherSuffixMapping.entrySet()) {
                        if (entry.getKey().matches(file)) {
                            configFileTypePathsMapping.get(entry.getValue()).add(file);
                        }
                    }
                    if (file.toString().endsWith(".jar")) {
                        PomDependency pomDependency = new PomDependency();
                        report.setFramework("java");
                        String fileName = file.getFileName().toString();
                        String substring = fileName.substring(0, fileName.indexOf("."));
                        int i = substring.lastIndexOf("-");
                        pomDependency.setType(2);
                        if (i > -1) {
                            pomDependency.setArtifactId(substring.substring(0, i));
                            pomDependency.setVersion(substring.substring(i));
                        } else {
                            pomDependency.setArtifactId(substring);
                        }
                        jarFiles.add(pomDependency);
                    }
                    if (file.toString().endsWith(".js")) {
                        PomDependency pomDependency = new PomDependency();
                        report.setFramework("javascript");
                        String fileName = file.getFileName().toString();
                        int i = fileName.lastIndexOf(".");
                        int i1 = fileName.indexOf("-");
                        pomDependency.setType(1);
                        if (i1 > -1) {
                            pomDependency.setArtifactId(fileName.substring(0, i1));
                            pomDependency.setVersion(fileName.substring(i1 + 1, i));
                        } else {
                            pomDependency.setArtifactId(fileName.substring(0, fileName.indexOf(".")));
                        }
                        jsFiles.add(pomDependency);
                    }
                    List<String> allLines;
                    try {
                        allLines = Files.readAllLines(file);
                    } catch (MalformedInputException e) {
                        return FileVisitResult.CONTINUE;
                    }
                    lineNum += allLines.size();

                    //国产
                    for (int i = 0; i < allLines.size(); i++) {
                        String s = allLines.get(i);
                        for (Rule rule : ruleList) {
                            if (s.indexOf(rule.getTarget()) > -1) {
                             //   warnList.add(new Warn(file.toString(), i + 1, rule.getTarget()));
                            }
                        }
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    return FileVisitResult.CONTINUE;
                }
            });
            //    WebSocketServer.sendIn(uuid,TimeUtil.getNowDate() + " 项目评估：" + "读取项目文件结束");
        } catch (IOException e) {
            e.printStackTrace();
        }
        return analysis();
    }

    //基本信息
    @Transactional
    public Report analysis() {
      //  report.setFileNum(fileNum);
       // report.setLineNum(lineNum);
        //report.setLanguage(Report.Language.ONLYVIEW);
        Report.Language most = null;
        int maxnum = 0;
        for (Map.Entry<String, Counter> entry : languageMatchMap.entrySet()) {
            if (entry.getValue().getNumber() > maxnum) {
                most = suffixLanguageMapping.get(entry.getKey());

                maxnum = entry.getValue().getNumber();
            }
        }
      //  report.setLanguage(most == null ? Report.Language.UNKNOW : most);
       // report.setFramework(most == null ? Report.Language.UNKNOW.name() : most.name());
        //report.setBackend(jarFiles);
        //report.setFrontend(jsFiles);
        //report.setCreateTime(new Date());
        /**
         * 对于具体的配置文件对应的处理方法
         * 增加要处理的文件类型需要在这里增加相应的处理方法
         */
        //后端依赖
        ArrayList<PomDependency> backend = new ArrayList<>();
        //前端依赖
        ArrayList<PomDependency> frontend = new ArrayList<>();

        for (Map.Entry<String, List<Path>> entry : configFileTypePathsMapping.entrySet()) {
            //System.out.println(entry.getKey());
            switch (entry.getKey()) {
                /**
                 * 配置文件的一个类型，xml文件
                 */
                case "xml":
                    //System.out.println(entry.getValue().size());
                    for (Path path : entry.getValue()) {
                        /**
                         * 对于maven工程，可以在maven配置文件中得到一这些信息
                         *      编译方式打包方式和打好的包的路径
                         */
                        if (path.getFileName().endsWith("pom.xml")) {
           //                 report.setDependenceManagement(Report.DependenceManagement.MAVEN);
             //               report.setCompileFilePath(path.toString());
                            //解析pom文件
                            ProjectPom projectPom = AnalysisFile.analysisPom(path);
                            //设置依赖
                            for (PomDependency dependency : projectPom.getDependencies()) {
                                dependency.setType(2);
                            }
                            for (Properties property : projectPom.getProperties()) {
                                versionMap.put(property.getName(),property.getValue());
                            }
                            jarFiles.addAll(projectPom.getDependencies());
                        }

                    }
                    break;
                /**
                 * properties文件
                 *
                 */
                case "gradle":
                    for (Path path : entry.getValue()) {
                        /**
                         * 对于maven工程，可以在maven配置文件中得到一这些信息
                         *      编译方式打包方式和打好的包的路径
                         */
                        if (path.getFileName().endsWith("build.gradle")) {
             //               report.setDependenceManagement(Report.DependenceManagement.GRADLE);
               //             report.setCompileFilePath(path.toString());
                            //解析pom文件
                            List<PomDependency> pomDependencies = AnalysisFile.analysisGradle(path);
                            //设置依赖
                            for (PomDependency dependency : pomDependencies) {
                                dependency.setType(2);
                            }
                            jarFiles.addAll(pomDependencies);
                        }
                    }
                case "json":
                    for (Path path : entry.getValue()) {
                        if ("package.json".equals(path.getFileName())) {
                            VueDep json = FileUtil.parseJsonFile(path.toString());
                            Map<String, String> dependencies = json.getDependencies();
                            for (String s : dependencies.keySet()) {
                                PomDependency pomDependency = new PomDependency();
                                pomDependency.setArtifactId(s);
                                pomDependency.setVersion(dependencies.get(s));
                                pomDependency.setType(1);
                                frontend.add(pomDependency);
                            }
                        }
                    }
                    break;
                default:
            }
        }

//        if (report.getLanguage() == Report.Language.JAVA) {
//            String tips = "该项目采用java语言开发，使用了主流框架，未见生僻插件，适配无明显难点。预计适配工作量较小，适配成功率高";
//            // TODO: 2020-02-16 人工估计工作量，一天200行代码
//        }
      //  List<PomDependency> backend1 = report.getBackend();
//        for (PomDependency pomDependency : backend1) {
//            String version = pomDependency.getVersion();
//            String versionString = version.replaceAll("\\$", "").replaceAll("\\{", "").replaceAll("}", "");
//            if(versionMap.containsKey(versionString)){
//                pomDependency.setVersion(versionMap.get(versionString));
//            }
//        }
//        report.setWarnList(warnList);
        return report;
    }

    public class Counter {
        private int i = 0;

        public void plus() {
            i++;
        }

        public int getNumber() {
            return i;
        }

        public void reset() {
            i = 0;
        }
    }

    public class YmlConfig {
        private String springDatasourceDriverClassName;
    }


    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Warn {
        private String path;
        private int line;
        private String keyWord;
        private String suggest;
    }

}
