package com.zjty.project.inspect.service.impl;

import com.zjty.project.inspect.dao.RuleDao;
import com.zjty.project.inspect.entity.CustomPage;
import com.zjty.project.inspect.entity.ExcelDataVo;
import com.zjty.project.inspect.entity.Rule;
import com.zjty.project.inspect.entity.RuleQo;
import com.zjty.project.inspect.service.RuleService;
import com.zjty.project.inspect.service.TechnologyService;
import com.zjty.project.inspect.utils.ExcelReader;
import com.zjty.project.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;

/**
 * 规则库
 *
 * @author Mcj
 * @date 2020-02-26 15:05
 */

@Service
@Transactional(rollbackFor = Exception.class)
public class RuleServiceImpl implements RuleService {

    /**
     * 规则库dao
     */
    @Autowired
    private RuleDao ruleDao;
    @Autowired
    private TechnologyService technologyService;

    @Override
    public void test() {
        List<ExcelDataVo> excelDataVos = ExcelReader.readExcel("C:\\Users\\Jey\\Desktop\\生成代码\\规则表.xlsx");
        System.out.println(excelDataVos);
        HashSet<String> set=new HashSet();
        for (ExcelDataVo excelDataVo : excelDataVos) {
            List<Rule> rules = ruleDao.findAllByTarget(excelDataVo.getName());
            if(rules==null||rules.size()==0){
                set.add(excelDataVo.getName());
            }
        }
        for (String s : set) {
            Rule rule=new Rule();
            rule.setTechnologyName("国产化依赖(支持)");
            rule.setTechnologyId("967a8bd0-eb27-4fb6-ba1d-1bca21cea5b4");
            rule.setSuffix("java");
            rule.setTarget(s);
            rule.setId(UUIDUtil.getUUID());
            ruleDao.save(rule);
        }

    }

    /**
     * 新增规则
     *
     * @param ruleQo 规则封装类
     */
    @Override
    public void addRule(RuleQo ruleQo) {
        List<String> suffixes = ruleQo.getSuffix();
        for (String suffix : suffixes) {
            Rule rule1 = ruleDao.findByTargetAndSuffixEqualsAndTechnologyIdEquals(ruleQo.getTarget(), suffix,ruleQo.getTechnologyId());
            if (rule1 != null) {
                return;
            }
            Rule rule = new Rule();
            rule.setTarget(ruleQo.getTarget());
            rule.setSuffix(suffix);
            rule.setTechnologyId(ruleQo.getTechnologyId());
            rule.setTechnologyName(ruleQo.getTechnologyName());
            rule.setId(UUIDUtil.getUUID());
            ruleDao.save(rule);
        }

    }

    /**
     * 条件查询+分页
     *
     * @param whereMap
     * @param page
     * @param size
     * @return
     */
    @Override
    public CustomPage<RuleQo> findSearch(Map whereMap, int page, int size) {
        Specification<Rule> specification = createSpecification(whereMap);
        PageRequest pageRequest = PageRequest.of(page - 1, size);
        Page<Rule> all = ruleDao.findAll(specification, pageRequest);
        List<RuleQo> ruleQos = data2RuleQo(all.getContent());

        CustomPage<RuleQo> myCustomPage = new CustomPage<>();
        myCustomPage.setTotalPage(all.getTotalPages());
        myCustomPage.setTotalElement((int)all.getTotalElements());
        myCustomPage.setObjects(ruleQos);
        return myCustomPage;
    }

    /**
     * 动态条件构建
     *
     * @param searchMap
     * @return
     */
    private Specification<Rule> createSpecification(Map searchMap) {

        return new Specification<Rule>() {

            @Override
            public Predicate toPredicate(Root<Rule> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<Predicate>();
                if (searchMap.get("technologyName") != null && !"".equals(searchMap.get("technologyName"))) {
                    predicateList.add(cb.like(root.get("technologyName").as(String.class), "%" + (String) searchMap.get("technologyName") + "%"));
                }
                if (searchMap.get("target") != null && !"".equals(searchMap.get("target"))) {
                    predicateList.add(cb.like(root.get("target").as(String.class), "%" + (String) searchMap.get("target") + "%"));
                }
                return cb.and(predicateList.toArray(new Predicate[predicateList.size()]));

            }
        };

    }

    @Override
    public void upRule(RuleQo ruleQo) {
        RuleQo oldRule = ruleQo.getOldRule();
        List<Rule> rules = ruleDao.findAllByTechnologyIdEqualsAndTargetEquals(oldRule.getTechnologyId(), oldRule.getTarget());
        for (Rule rule : rules) {
            ruleDao.deleteById(rule.getId());
        }
        addRule(ruleQo);
    }

    @Override
    public void deleteById(String id) {
        ruleDao.deleteById(id);
    }

    @Override
    @Modifying
    public void deleteRule(RuleQo ruleQo) {
        List<Rule> rules = ruleDao.findAllByTechnologyIdEqualsAndTargetEquals(ruleQo.getTechnologyId(), ruleQo.getTarget());
        for (Rule rule : rules) {
            ruleDao.deleteById(rule.getId());
        }
    }

    /**
     * 查询所有规则
     * 规则唯一 关键字+后缀+技术id
     * target:xx,suffix:js:tech:gjfgj
     * target:xx,suffix:html:tech:gjfgj
     * target:xx,suffix:css:tech:gjfgj
     *
     * target:xx,suffix:html,css,js:tech:gjfgj
     * @return RuleVoList
     */
    @Override
    public List<RuleQo> findAll() {
        List<Rule> rules = ruleDao.findAll();
        List<RuleQo> ruleQos = data2RuleQo(rules);
        return ruleQos;
    }

    @Override
    public List<RuleQo> findByName(String name) {
        List<Rule> rules = ruleDao.findAllByTargetLike("%" + name + "%");
        if (rules != null && rules.size() > 0) {
            return data2RuleQo(rules);
        }
        return null;
    }

    private List<RuleQo> data2RuleQo(List<Rule> rules){
        HashMap<String, RuleQo> hashMap = new HashMap<>();
        for (Rule rule : rules) {
            if(!hashMap.containsKey(rule.getTarget()+rule.getTechnologyId())){
                RuleQo ruleQo = new RuleQo();
                ruleQo.setId(rule.getId());
                ruleQo.setTarget(rule.getTarget());
                ruleQo.setTechnologyId(rule.getTechnologyId());
                ruleQo.setTechnologyName(rule.getTechnologyName());
                ArrayList<String> suffix = new ArrayList<>();
                suffix.add(rule.getSuffix());
                ruleQo.setSuffix(suffix);
                hashMap.put(rule.getTarget()+rule.getTechnologyId(),ruleQo);

            }else{
                RuleQo ruleQo = hashMap.get(rule.getTarget() + rule.getTechnologyId());
                ruleQo.getSuffix().add(rule.getSuffix());
            }
        }
        return new ArrayList<>(hashMap.values());
    }
}
