package com.zjty.project.state.config.websocket;

import com.zjty.project.state.base.container.DataContainer;
import com.zjty.project.state.utils.ExceptionMessage;
import lombok.Data;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.io.IOException;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * @program: liujie-cloud
 * @description: WebSocket服务类
 * @author: LiuJie
 * @create: 2019-05-17 10:08
 **/

@ServerEndpoint("/websocket/{sid}")
@Component
@Data
public class WebSocketServer {
    static Logger log= LoggerFactory.getLogger(WebSocketServer.class);
    @Autowired
    private DataContainer dataContainer;
    /**
     * 静态变量，用来记录当前在线连接数。应该把它设计成线程安全的。
     */
    private static int onlineCount = 0;
    /**
     * concurrent包的线程安全Set，用来存放每个客户端对应的MyWebSocket对象。
     */
    public static CopyOnWriteArraySet<WebSocketServer> webSocketSet = new CopyOnWriteArraySet<WebSocketServer>();

    /**
     * 与某个客户端的连接会话，需要通过它来给客户端发送数据
     */
    private Session session;

    /**
     * 接收sid
     */
    private String sid="";

    /**
     * 连接建立成功调用的方法*/
    @OnOpen
    public void onOpen(Session session,@PathParam("sid") String sid) {
        this.session = session;
        //加入set中
        webSocketSet.add(this);
        //在线数加1
        addOnlineCount();
        log.info("有新窗口开始监听:"+sid+",当前在线人数为" + getOnlineCount());
        this.sid=sid;
        try {
            sendMessage("连接成功");
        } catch (IOException e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            log.error("websocket IO异常:"+info);
        }
    }

    /**
     * 连接关闭调用的方法
     */
    @OnClose
    public void onClose() {
        //从set中删除
        webSocketSet.remove(this);
        //在线数减1
        subOnlineCount();
        log.info("有一连接关闭！当前在线人数为" + getOnlineCount());
    }

    /**
     * 收到客户端消息后调用的方法
     *
     * @param message 客户端发送过来的消息*/
    @OnMessage
    public void onMessage(String message, Session session) {
        log.info("收到来自窗口"+sid+"的信息:"+message);
        /**
         * 群发消息
         */
        for (WebSocketServer item : webSocketSet) {
            try {
                item.sendMessage(message);
            } catch (Exception e) {
                String info = ExceptionMessage.getStackTraceInfo(e);
                log.error("发生错误："+info);
            }
        }
    }

    /**
     *
     * @param session
     * @param error
     */
    @OnError
    public void onError(Session session, Throwable error) {
        log.error("发生错误");
        String error1 = ExceptionMessage.getError(error);
        log.info("error"+error1);
        error.printStackTrace();
    }
   /**
   * @Description: 实现服务器主动推送
   * @Param:
   * @return:
   * @Author: LiuJie
   * @Date: 2019/5/31
   */
    public void sendMessage(String message) throws IOException {
        this.session.getBasicRemote().sendText(message);
    }

    /**
     * @Description: 实现服务器主动向所有的客户端推送数据
     * @Param:
     * @return:
     * @Author: LiuJie
     * @Date: 2019/5/31
     */
    public void sendMessageTo(String message) throws IOException {
        for (WebSocketServer item : webSocketSet) {
            try {
                item.session.getBasicRemote().sendText(message);
            } catch (Exception e) {
                String info = ExceptionMessage.getStackTraceInfo(e);
                log.error("发生错误："+info);
            }
        }
    }
    
    /**
     * 群发自定义消息
     * */
    public void sendInfo(String message,@PathParam("sid") String sid) throws IOException {
//        log.info("推送消息到窗口"+sid+"，推送内容:"+message);
        for (WebSocketServer item : webSocketSet) {
            try {
                /**
                 * 这里可以设定只推送给这个sid的，为null则全部推送
                 */
                if(sid==null) {
                    item.sendMessage(message);
                }else if(item.sid.equals(sid)){
                    item.sendMessage(message);
                }
            } catch (IOException e) {
                String info = ExceptionMessage.getStackTraceInfo(e);
                log.error("error:"+info);
                continue;
            }
        }
    }

    public static synchronized int getOnlineCount() {
        return onlineCount;
    }

    public static synchronized void addOnlineCount() {
        WebSocketServer.onlineCount++;
    }

    public static synchronized void subOnlineCount() {
        WebSocketServer.onlineCount--;
    }


    @Override
    public boolean equals(Object o) {
        if (this == o){
            return true;
        }
        if (o == null || getClass() != o.getClass()){
            return false;
        }
        if (!super.equals(o)){
            return false;
        }

        WebSocketServer that = (WebSocketServer) o;

        if (session != null ? !session.equals(that.session) : that.session != null){
            return false;
        }
        return sid != null ? sid.equals(that.sid) : that.sid == null;
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + (session != null ? session.hashCode() : 0);
        result = 31 * result + (sid != null ? sid.hashCode() : 0);
        return result;
    }

}
