package com.zjty.project.state.server.controller;

import com.zjty.project.state.base.exception.MyException;
import com.zjty.project.state.base.response.ResponseCode;
import com.zjty.project.state.base.response.ServerResponse;
import com.zjty.project.state.server.pojo.Unit;
import com.zjty.project.state.server.service.UnitService;
import com.zjty.project.state.utils.BaseRequest;
import com.zjty.project.state.utils.ExceptionMessage;
import com.zjty.project.state.utils.PageResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.List;

/**
 * @program: demo
 * @description:
 * @author: LiuJie
 * @create: 2020-02-13 16:04
 **/
@Api(description = "UnitController单位接口")
@RestController
@RequestMapping("/unit")
public class UnitController {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private UnitService unitService;


    @ApiOperation(value = "文件导入", notes = "文件导入，导入Excel表格文件，生成基础单位名录")
    @RequestMapping(value = "/upload", method = RequestMethod.POST)
    public ServerResponse importExcel(@RequestParam("file") MultipartFile file) throws IOException, Exception{
        try {
            unitService.importExcel(file);
            return ServerResponse.success("导入成功。");
        } catch (Exception e) {
            logger.error("文件导入单位:{}",ExceptionMessage.getStackTraceInfo(e));
            throw new MyException(ResponseCode.FILE_IS_ERROR);
        }
    }

    @ApiOperation(value = "新建单位", notes = "新建单位")
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ServerResponse add(@RequestBody Unit unit) throws Exception{
        try {
            unitService.add(unit);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("新建单位失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
        return ServerResponse.success("新建单位成功。");
    }



    @ApiOperation(value = "获取所有单位列表", notes = "获取所有单位列表：查询所有单位")
    @RequestMapping(value = "/findAll", method = RequestMethod.GET)
    public ServerResponse findAll(){
        try {
            List<Unit> list= unitService.findAll();
            return ServerResponse.success(list);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取所有单位列表失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }

    @ApiOperation(value = "获取准入单位列表", notes = "获取准入单位列表：查询所有准入单位")
    @RequestMapping(value = "/findAccess", method = RequestMethod.GET)
    public ServerResponse findAccess(){
        try {
            List<Unit> list= unitService.findAccess();
            return ServerResponse.success(list);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取准入单位列表失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }

    @ApiOperation(value = "根据单位id查询单位信息", notes = "根据单位id查询单位信息")
    @RequestMapping(value = "/get/{id}", method = RequestMethod.GET)
    public ServerResponse get(@PathVariable("id") Integer id){
        try {
             Unit unit = unitService.findById(id);
             if (unit!=null){
                 return ServerResponse.success(unit);
             }else {
                 return ServerResponse.error("单位不存在。");
             }
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("根据单位id查询单位信息失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }



    @ApiOperation(value = "单位分页查询", notes = "单位分页查询：根据条件查询单位列表")
    @RequestMapping(value = "/search", method = RequestMethod.POST)
    public ServerResponse search(@RequestBody BaseRequest<Unit> baseRequest) throws Exception{
        try {
            PageResult<Unit> pageResult= unitService.search(baseRequest);
            return ServerResponse.success(pageResult);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            return ServerResponse.error("单位分页查询失败。");
        }
    }

    @ApiOperation(value = "准入单位分页查询", notes = "准入单位分页查询：根据条件查询准入单位列表")
    @RequestMapping(value = "/searchAccess", method = RequestMethod.POST)
    public ServerResponse searchAccess(@RequestBody BaseRequest<Unit> baseRequest) throws Exception{
        try {
            Unit unit = baseRequest.getT();
            if (unit==null){
                unit = new Unit();
            }
            unit.setStatus(1);
            PageResult<Unit> pageResult= unitService.search(baseRequest);
            return ServerResponse.success(pageResult);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            return ServerResponse.error("准入单位分页查询失败。");
        }
    }



    @ApiOperation(value = "根据单位类别，获取该类别单位信息", notes = "根据单位类别，获取该类别单位信息")
    @RequestMapping(value = "/findByType/{type}", method = RequestMethod.GET)
    public ServerResponse findByType(@PathVariable("type") Integer type){
        try {
            List<Unit> unitList = unitService.findByType(type);
            return ServerResponse.success(unitList);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("根据单位类别，获取该类别单位信息失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }





}
