package com.zjty.project.inspect.service.impl;

import com.zjty.project.inspect.dao.RuleCollectionDao;
import com.zjty.project.inspect.dao.RuleDao;
import com.zjty.project.inspect.entity.*;
import com.zjty.project.inspect.service.RuleService;
import com.zjty.project.inspect.utils.UUIDUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.Predicate;
import java.util.*;

/**
 * 规则库
 *
 * @author Mcj
 * @date 2020-02-26 15:05
 */

@Service
@Transactional(rollbackFor = Exception.class)
public class RuleServiceImpl implements RuleService {

    /**
     * 规则库dao
     */
    @Autowired
    private RuleDao ruleDao;

    @Autowired
    private RuleCollectionDao ruleCollectionDao;

    /**
     * 新增规则
     *
     * @param ruleQo 规则封装类
     */
    @Override
    public void addRule(RuleQo ruleQo) {
        List<String> suffixes = ruleQo.getSuffix();
        for (String suffix : suffixes) {
            Rule rule1 = ruleDao.findByTargetAndSuffixEqualsAndTechnologyIdEquals(ruleQo.getTarget(), suffix,ruleQo.getTechnologyId());
            if (rule1 != null) {
                continue;
            }
            Rule rule = new Rule();
            rule.setTarget(ruleQo.getTarget());
            rule.setSuffix(suffix);
            rule.setTechnologyId(ruleQo.getTechnologyId());
            rule.setTechnologyName(ruleQo.getTechnologyName());
            rule.setId(UUIDUtil.getUUID());
            ruleDao.save(rule);
        }
        RuleCollection ruleCollection = dataQo2RuleCollection(ruleQo);
        ruleCollectionDao.save(ruleCollection);

    }

    @Override
    public void addRule(List<Rule> rules) {
        ruleDao.saveAll(rules);
        List<RuleCollection> ruleCollections = dataList2RuleCollection(rules);
        ruleCollectionDao.saveAll(ruleCollections);
    }

    /**
     * 条件查询+分页
     *
     * @param whereMap
     * @param page
     * @param size
     * @return
     */
    @Override
    public CustomPage<RuleCollection> findSearch(Map whereMap, int page, int size) {
        Specification<RuleCollection> specification = createSpecification(whereMap);
        Sort sort = new Sort(Sort.Direction.DESC, "updateDate");
        PageRequest pageRequest = PageRequest.of(page - 1, size, sort);
        Page<RuleCollection> ruleCollections = ruleCollectionDao.findAll(specification, pageRequest);

        CustomPage<RuleCollection> customPage = new CustomPage<>();
        customPage.setTotalPage(ruleCollections.getTotalPages());
        customPage.setTotalElement((int)ruleCollections.getTotalElements());
        customPage.setObjects(ruleCollections.getContent());
        return customPage;
    }

    @Override
    public List<Rule> findAllByTechnologyIdIn(List<String> technologyIds) {
        return ruleDao.findAllByTechnologyIdIn(technologyIds);
    }

    /**
     * 动态条件构建
     *
     * @param searchMap
     * @return
     */
    private Specification<RuleCollection> createSpecification(Map searchMap) {

        return (root, query, cb) -> {
            List<Predicate> predicateList = new ArrayList<>();
            if (searchMap.get("technologyName") != null && !"".equals(searchMap.get("technologyName"))) {
                predicateList.add(cb.like(root.get("technologyName").as(String.class), "%" + searchMap.get("technologyName") + "%"));
            }
            if (searchMap.get("target") != null && !"".equals(searchMap.get("target"))) {
                predicateList.add(cb.like(root.get("target").as(String.class), "%" + searchMap.get("target") + "%"));
            }
            return cb.and(predicateList.toArray(new Predicate[predicateList.size()]));

        };

    }

    @Override
    public void upRule(RuleQo ruleQo) {
        RuleQo oldRule = ruleQo.getOldRule();
        List<Rule> rules = ruleDao.findAllByTechnologyIdEqualsAndTargetEquals(oldRule.getTechnologyId(), oldRule.getTarget());
        RuleCollection ruleCollection = ruleCollectionDao.findAllByTechnologyIdEqualsAndTargetEquals(oldRule.getTechnologyId(), oldRule.getTarget());
        for (Rule rule : rules) {
            ruleDao.deleteById(rule.getId());
        }
        if(ruleCollection!=null){
            ruleCollectionDao.deleteById(ruleCollection.getId());
        }
        addRule(ruleQo);
    }


    @Override
    @Modifying
    public void deleteRule(RuleQo ruleQo) {
        List<Rule> rules = ruleDao.findAllByTechnologyIdEqualsAndTargetEquals(ruleQo.getTechnologyId(), ruleQo.getTarget());
        for (Rule rule : rules) {
            ruleDao.deleteById(rule.getId());
        }
        RuleCollection ruleCollection = ruleCollectionDao.findAllByTechnologyIdEqualsAndTargetEquals(ruleQo.getTechnologyId(), ruleQo.getTarget());
        if(ruleCollection!=null){
            ruleCollectionDao.deleteById(ruleCollection.getId());
        }

    }

    /**
     * 查询所有规则
     * 规则唯一 关键字+后缀+技术id
     * target:xx,suffix:js:tech:gjfgj
     * target:xx,suffix:html:tech:gjfgj
     * target:xx,suffix:css:tech:gjfgj
     *
     * target:xx,suffix:html,css,js:tech:gjfgj
     * @return RuleVoList
     */
    @Override
    public List<RuleCollection> findAll() {
        List<RuleCollection> rules = ruleCollectionDao.findAll();
        return rules;
    }

    @Override
    public List<Rule> findRules() {
        return ruleDao.findAll();
    }

    @Override
    public List<RuleCollection> findByName(String name) {
        List<RuleCollection> rules = ruleCollectionDao.findAllByTargetLike("%" + name + "%");
        return rules;
    }

    private RuleCollection dataQo2RuleCollection(RuleQo ruleQo){
        RuleCollection ruleCollection = new RuleCollection();
        ruleCollection.setTechnologyName(ruleQo.getTechnologyName());
        ruleCollection.setTechnologyId(ruleQo.getTechnologyId());
        ruleCollection.setTarget(ruleQo.getTarget());
        ruleCollection.setId(UUIDUtil.getUUID());
        StringBuilder stringBuilder = new StringBuilder();
        for (String suffix : ruleQo.getSuffix()) {
            if(stringBuilder.length()==0){
                stringBuilder.append(suffix);
            }else{
                stringBuilder.append(",").append(suffix);
            }
        }
        ruleCollection.setSuffix(stringBuilder.toString());
        return ruleCollection;
    }

    private List<RuleCollection> dataList2RuleCollection(List<Rule> rules){
        ArrayList<RuleCollection> ruleCollections = new ArrayList<>();
        HashMap<String, ArrayList<String>> map = new HashMap<>();
        HashMap<String, Rule> ruleMap = new HashMap<>();
        for (Rule rule : rules) {
            ruleMap.put(rule.getTarget()+":"+rule.getTechnologyId(),rule);
            if(map.containsKey(rule.getTarget()+":"+rule.getTechnologyId())){
                ArrayList<String> suffix = map.get(rule.getTarget()+":"+rule.getTechnologyId());
                suffix.add(rule.getSuffix());
            }else{
                ArrayList<String> suffix = new ArrayList<>();
                suffix.add(rule.getSuffix());
                map.put(rule.getTarget()+":"+rule.getTechnologyId(),suffix);
            }
        }

        for (String target : map.keySet()) {
            Rule rule = ruleMap.get(target);
            RuleCollection ruleCollection = new RuleCollection();
            ruleCollection.setTarget(rule.getTarget());
            ruleCollection.setTechnologyId(rule.getTechnologyId());
            ruleCollection.setTechnologyName(rule.getTechnologyName());
            ruleCollection.setId(UUIDUtil.getUUID());
            StringBuilder stringBuilder = new StringBuilder();
            for (String suffix : map.get(target)) {
                if(stringBuilder.length()==0){
                    stringBuilder.append(suffix);
                }else{
                    stringBuilder.append(",").append(suffix);
                }
            }
            ruleCollection.setSuffix(stringBuilder.toString());
            ruleCollections.add(ruleCollection);
        }
        return ruleCollections;
    }
}
