package com.zjty.project.state.server.controller;

import com.zjty.project.state.base.exception.MyException;
import com.zjty.project.state.base.response.ResponseCode;
import com.zjty.project.state.base.response.ServerResponse;
import com.zjty.project.state.server.pojo.Produce;
import com.zjty.project.state.server.service.ProduceService;
import com.zjty.project.state.utils.BaseRequest;
import com.zjty.project.state.utils.ExceptionMessage;
import com.zjty.project.state.utils.PageResult;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.util.List;

/**
 * @program: demo
 * @description:
 * @author: LiuJie
 * @create: 2020-02-13 16:04
 **/
@Api(description = "ProduceController产品接口")
@RestController
@RequestMapping("/produce")
public class ProduceController {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ProduceService produceService;


    @ApiOperation(value = "文件导入", notes = "文件导入，导入Excel表格文件，生成基础产品名录")
    @RequestMapping(value = "/upload", method = RequestMethod.POST)
    public ServerResponse importExcel(@RequestParam("file") MultipartFile file) throws IOException, Exception{
        try {
            produceService.importExcel(file);
            return ServerResponse.success("导入成功。");
        } catch (Exception e) {
            logger.error("文件导入产品:{}",ExceptionMessage.getStackTraceInfo(e));
            throw new MyException(ResponseCode.FILE_IS_ERROR);
        }
    }

    @ApiOperation(value = "新建产品", notes = "新建产品")
    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ServerResponse add(@RequestBody Produce produce) throws Exception{
        try {
            produceService.add(produce);
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("新建产品失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
        return ServerResponse.success("新建产品成功。");
    }

    @ApiOperation(value = "产品信息修改", notes = "产品信息修改")
    @RequestMapping(value = "/update", method = RequestMethod.PUT)
    public ServerResponse update(@RequestBody Produce produce) throws Exception{
        try {
            produceService.update(produce);
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("产品信息修改失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
        return ServerResponse.success("产品信息修改成功。");
    }



    @ApiOperation(value = "获取所有产品列表", notes = "获取所有产品列表：查询所有产品")
    @RequestMapping(value = "/findAll", method = RequestMethod.GET)
    public ServerResponse findAll(){
        try {
            List<Produce> list= produceService.findAll();
            return ServerResponse.success(list);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取所有产品列表失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }



    @ApiOperation(value = "获取准入产品列表", notes = "获取准入产品列表：查询所有准入产品")
    @RequestMapping(value = "/findAccess", method = RequestMethod.GET)
    public ServerResponse findAccess(){
        try {
            List<Produce> list= produceService.findAccess();
            return ServerResponse.success(list);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取准入产品列表失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }

    @ApiOperation(value = "产品分页查询", notes = "产品分页查询：根据条件查询角色列表")
    @RequestMapping(value = "/search", method = RequestMethod.POST)
    public ServerResponse search(@RequestBody BaseRequest<Produce> baseRequest) throws Exception{
        try {
            PageResult<Produce> pageResult= produceService.search(baseRequest);
            return ServerResponse.success(pageResult);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("产品分页查询失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }

    @ApiOperation(value = "准入产品分页查询", notes = "准入产品分页查询：根据条件查询角色列表")
    @RequestMapping(value = "/searchAccess", method = RequestMethod.POST)
    public ServerResponse searchAccess(@RequestBody BaseRequest<Produce> baseRequest) throws Exception{
        try {
            Produce produce = baseRequest.getT();
            if (produce==null){
                produce = new Produce();
            }
            produce.setStatus(1);
            PageResult<Produce> pageResult= produceService.search(baseRequest);
            return ServerResponse.success(pageResult);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("准入产品分页查询失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }


    @ApiOperation(value = "根据技术路线，获取技术路线产品组合", notes = "根据技术路线，获取技术路线产品组合")
    @RequestMapping(value = "/findByTech/{techId}", method = RequestMethod.GET)
    public ServerResponse findByTech(@PathVariable("techId") Integer techId){
        try {
            List<Produce> list= produceService.findByTech(techId);
            return ServerResponse.success(list);
        }catch (MyException myException) {
            throw myException;
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取技术路线产品组合失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }

    @ApiOperation(value = "根据产品类型，获取类型产品组合", notes = "根据产品类型，获取类型产品组合")
    @GetMapping("/findByForm/{form}")
    public ServerResponse findByForm(@PathVariable("form") Integer form){
        try {
            List<Produce> list= produceService.findByForm(form);
            return ServerResponse.success(list);
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("获取类型产品组合失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }


    @ApiOperation(value = "根据软硬件分类获产品取列表", notes = "根据软硬件分类获产品取列表")
    @GetMapping("/findByType/{type}")
    public ServerResponse findByType(@PathVariable("type") Integer type){
        try {
            List<Produce> list= produceService.findByType(type);
            return ServerResponse.success(list);
        } catch (Exception e) {
            String info = ExceptionMessage.getStackTraceInfo(e);
            logger.error("根据软硬件分类获产品取列表失败:{}",info);
            throw new MyException(ResponseCode.SERVER_ERROR);
        }
    }



}
