package com.zjty.project.state.server.service;

import com.alibaba.fastjson.JSON;
import com.github.wenhao.jpa.Specifications;
import com.zjty.project.state.base.exception.MyException;
import com.zjty.project.state.base.response.ResponseCode;
import com.zjty.project.state.server.dao.UserDao;
import com.zjty.project.state.server.pojo.user.User;
import com.zjty.project.state.utils.BaseRequest;
import com.zjty.project.state.utils.ExceptionMessage;
import com.zjty.project.state.utils.PageResult;
import com.zjty.project.state.utils.UUIDUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @program: state
 * @description:
 * @author: LiuJie
 * @create: 2020-02-26 11:14
 **/
@Service
public class UserService {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${upload.location}")
    private String uploadPath;
    @Value("${server.port}")
    private String port;
    @Value("${host.ip}")
    private String localHost;

    @Autowired
    private UserDao userDao;
    @Autowired
    private UnitService unitService;
    @Autowired
    private RoleService roleService;


    @Transactional(rollbackFor = Exception.class)
    public void add(User user) {
        userDao.save(user);
    }

    public PageResult<User> search(BaseRequest<User> baseRequest) {
        Specification<User> specifications = null;
        User user = baseRequest.getT();
        logger.info("user:{}",JSON.toJSONString(user));
        if (user==null){
            specifications = Specifications.<User>and()
                    .build();
        }else {
            specifications = Specifications.<User>and()
                    .like(user.getName()!=null && !Objects.equals(user.getName(), ""),"name","%"+user.getName()+"%")
                    .eq(user.getRoleId()!=null && !Objects.equals(user.getRoleId(), ""),"roleId",user.getRoleId())
                    .eq(user.getUnitId()!=null && !Objects.equals(user.getUnitId(), ""),"unitId",user.getUnitId())
                    .build();
        }
        Integer page = baseRequest.getPage();
        Integer pageSize = baseRequest.getPageSize();
        if (page==null || page<=0){
            page = 1;
        }
        if (pageSize==null || pageSize<=0){
            pageSize = 10;
        }
        Pageable pageable = PageRequest.of(page-1, pageSize, Sort.Direction.DESC, "id");
        Long count = userDao.count(specifications);
        List<User> userList = userDao.findAll(specifications,pageable).getContent();
        logger.info("userList:{}", JSON.toJSONString(userList));
        for (User u:userList){
            Integer unitId = u.getUnitId();
            String unitName = unitService.getName(unitId);
            Integer roleId = u.getRoleId();
            String roleName = roleService.getName(roleId);
            u.setUnitName(unitName);
            u.setRoleName(roleName);
        }
        return new PageResult<User>(page,pageSize,count,userList);
    }

    public User login(String userName) {
        //String userName = user.getUserName();
        List<User> list = userDao.findAllByUserName(userName);
        if (list!=null&&list.size()>0){
            return list.get(0);
        }else {
            return null;
        }
    }

    public User findById(Integer userId) {
        if (userId==null){
            return null;
        }
        Optional<User> optional = userDao.findById(userId);
        return optional.isPresent()?optional.get():null;
    }

    @Transactional(rollbackFor = Exception.class)
    public void upload(Integer fileId,MultipartFile multipartFile) {
        java.io.File realPathFile = new java.io.File(uploadPath);
        if (!realPathFile.exists()) {
            realPathFile.mkdirs();
        }
        User user = findById(fileId);
        if (user!=null){
            if (multipartFile != null ) {
                logger.info("照片文件不为空");
                if (!multipartFile.isEmpty()) {
                    String name = multipartFile.getOriginalFilename();
                    String[] split = StringUtils.split(name, ".");
                    String fileType = split[split.length - 1];
                    String uuidFileName = UUIDUtils.getUUIDFileName(multipartFile.getOriginalFilename()) +"."+fileType;
                    try {
                        InputStream inputStream = multipartFile.getInputStream();
                        Files.copy(inputStream, Paths.get(uploadPath, uuidFileName), StandardCopyOption.REPLACE_EXISTING);
                        inputStream.close();
                        String photo = "http://" + localHost + ":" + port + "/" + uploadPath + uuidFileName;
                        user.setPath(photo);
                        userDao.saveAndFlush(user);
                    } catch (IOException e) {
                        String info = ExceptionMessage.getStackTraceInfo(e);
                        logger.error("照片文件保存失败:" + info);
                    }
                }
            } else {
                logger.error("照片文件为空");
            }
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void update(User user) {
        User u = findById(user.getId());
        if (u!=null){
            u.setName(user.getName());
            u.setNumber(user.getNumber());
            u.setRoleId(user.getRoleId());
            u.setDeleted(user.getDeleted());
            u.setUnitId(user.getUnitId());
            u.setSex(user.getSex());
            userDao.saveAndFlush(u);
        }
    }

    public String getPath(MultipartFile multipartFile) {
        java.io.File realPathFile = new java.io.File(uploadPath);
        if (!realPathFile.exists()) {
            realPathFile.mkdirs();
        }
        String photo = null;
        if (multipartFile != null ) {
            logger.info("照片文件不为空");
            if (!multipartFile.isEmpty()) {
                String name = multipartFile.getOriginalFilename();
                String[] split = StringUtils.split(name, ".");
                String fileType = split[split.length - 1];
                String uuidFileName = UUIDUtils.getUUIDFileName(multipartFile.getOriginalFilename()) +"."+fileType;
                try {
                    InputStream inputStream = multipartFile.getInputStream();
                    Files.copy(inputStream, Paths.get(uploadPath, uuidFileName), StandardCopyOption.REPLACE_EXISTING);
                    inputStream.close();
                    photo = "http://" + localHost + ":" + port + "/" + uploadPath + uuidFileName;
                } catch (IOException e) {
                    String info = ExceptionMessage.getStackTraceInfo(e);
                    logger.error("照片文件保存失败:" + info);
                }
            }
        } else {
            logger.error("照片文件为空");
            new MyException(ResponseCode.SERVER_ERROR);
        }
        return photo;
    }
}
