package com.zjty.automatedtesting.controller;

import com.google.common.collect.ImmutableMap;
import com.zjty.automatedtesting.pojo.report.ReportVo;
import com.zjty.automatedtesting.pojo.test.CaseVo;
import com.zjty.automatedtesting.service.SeleniumService;
import com.zjty.automatedtesting.service.CaseService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author C
 */
@SuppressWarnings("SpringJavaAutowiredFieldsWarningInspection")
@RequestMapping("/testcase")
@RestController
@Api(tags = "测试用例")
@CrossOrigin
public class TestCaseController {

    @Autowired
    SeleniumService seleniumService;

    @Autowired
    CaseService caseService;

    @ApiOperation(value = "提供测试用例数据.执行测试并返回测试结果报告.")
    @PostMapping(value = "/execute")
    public ResponseEntity<ReportVo> execute(@RequestBody CaseVo caseVo) {
        ReportVo execute = seleniumService.execute(caseVo);
        return ResponseEntity.ok(execute);
    }

    @ApiOperation(value = "根据测试系统id.执行改测试系统下的所有测试用例并返回测试结果报告的集合.")
    @GetMapping(value = "/execute/system/{systemId}")
    public ResponseEntity<List<ReportVo>> execute(@PathVariable String systemId) {
        List<CaseVo> caseVoList = caseService.findBySystemId(systemId);
        return ResponseEntity.ok(caseVoList.stream()
                .map(caseVo -> seleniumService.execute(caseVo))
                .collect(Collectors.toList()));
    }

    @ApiOperation(value = "新增测试用例.", notes = "不附带id")
    @PostMapping(value = "/create")
    public ResponseEntity<Map<String, String>> create(@RequestBody CaseVo caseVo) {
        caseService.create(caseVo);
        return ResponseEntity.ok(ImmutableMap.of("message", "success"));
    }

    @ApiOperation(value = "修改测试用例.", notes = "必须附带id")
    @PostMapping(value = "/update")
    public ResponseEntity<Map<String, String>> update(@RequestBody CaseVo caseVo) {
        caseService.update(caseVo);
        return ResponseEntity.ok(ImmutableMap.of("message", "success"));
    }

    @ApiOperation(value = "删除测试用例.")
    @DeleteMapping(value = "/delete/{id}")
    public ResponseEntity<Map<String, String>> findTestText(@PathVariable Integer id) {
        caseService.delete(id);
        return ResponseEntity.ok(ImmutableMap.of("message", "success"));
    }

    @ApiOperation(value = "获取全部测试用例.")
    @GetMapping(value = "/get")
    public ResponseEntity<List<CaseVo>> get() {
        return ResponseEntity.ok(caseService.findAll());
    }

    @ApiOperation(value = "根据测试系统id查询指定测试系统下的所有测试用例.")
    @GetMapping(value = "/get/system/{systemId}")
    public ResponseEntity<List<CaseVo>> getById(@PathVariable String systemId) {
        return ResponseEntity.ok(caseService.findBySystemId(systemId));
    }

    @ApiOperation(value = "获取单个测试用例.")
    @GetMapping(value = "/get/{id}")
    public ResponseEntity<CaseVo> getById(@PathVariable Integer id) {
        return ResponseEntity.ok(caseService.findById(id));
    }


}
