package com.zjty.automatedtesting.service;

import com.google.common.collect.Lists;
import com.zjty.automatedtesting.common.action.Assertion;
import com.zjty.automatedtesting.common.action.Browser;
import com.zjty.automatedtesting.pojo.report.Measure;
import com.zjty.automatedtesting.pojo.report.Report;
import com.zjty.automatedtesting.pojo.test.Step;
import com.zjty.automatedtesting.pojo.test.TestCaseVo;
import com.zjty.automatedtesting.util.CommonUtils;
import lombok.extern.slf4j.Slf4j;
import org.openqa.selenium.By;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.springframework.stereotype.Service;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.zjty.automatedtesting.common.action.Action.*;
import static com.zjty.automatedtesting.common.action.ByType.*;
import static java.util.Objects.isNull;

/**
 * @author C
 */
@Slf4j
@Service
public class SeleniumServiceImpl implements SeleniumService {

    @Override
    public Report execute(TestCaseVo testCase) {
        WebDriver driver = initDriver(testCase.getBrowser());
        driver.get(testCase.getUrl());
        return executeSteps(driver, testCase);
    }

    /**
     * 初始化driver
     *
     * @return driver
     */
    private WebDriver initDriver(String browser) {
        if (Objects.equals(browser, Browser.FIREFOX)) {
            System.setProperty("webdriver.firefox.driver", CommonUtils.FIRE_FOX_EXE);
            return new FirefoxDriver();
        } else if (Objects.equals(browser, Browser.CHROME)) {
            System.setProperty("webdriver.chrome.driver", CommonUtils.CHROME_EXE);
            return new ChromeDriver();
        } else if (Objects.equals(browser, Browser.IE)) {
            System.setProperty("webdriver.ie.driver", CommonUtils.IE_EXE);
            return new ChromeDriver();
        }else {
            throw new RuntimeException("该浏览器不存在：" + browser);
        }
    }

    private Report executeSteps(WebDriver driver, TestCaseVo testCase) {
        List<Measure> measures = Lists.newArrayList();
        List<Step> steps = testCase.getSteps().stream().sorted(Comparator.comparingInt(Step::getOrder)).collect(Collectors.toList());
        for (Step step : steps) {
            boolean success = false;
            String practice = "";
            String message;
            log.info("Executing Step:{}", step.getTitle());
            try {
                if (Objects.equals(step.getAction(), INPUT)) {
                    String value = isNull(step.getValue()) ? "" : step.getValue();
                    WebElement webElement = getWebElement(step.getType(), step.getKey(), driver);
                    webElement.sendKeys(value);
                } else if (Objects.equals(step.getAction(), CLICK)) {
                    WebElement webElement = getWebElement(step.getType(), step.getKey(), driver);
                    webElement.click();
                } else {
                    throw new RuntimeException("不匹配的操作类型：" + step.getAction());
                }
                if (Objects.equals(step.getAssertion(), Assertion.VALUE)) {
                    WebElement webElement = getWebElement(step.getType(), step.getKey(), driver);
                    practice = webElement.getAttribute("value");
                    if (Objects.equals(practice, step.getExpected())) {
                        success = true;
                        message = String.format("[%s] success.", step.getTitle());
                    } else {
                        message = String.format("[%s] failed.the value is incorrect. expected:%s  practice:[%s] ", step.getTitle(), step.getExpected(), practice);
                    }
                } else if (Objects.equals(step.getAssertion(), Assertion.TITLE)){
                    practice = driver.getTitle();
                    if (Objects.equals(practice, step.getExpected())) {
                        success = true;
                        message = String.format("[%s] success.", step.getTitle());
                    } else {
                        message = String.format("[%s] failed.the page is incorrect. expected:[%s]  practice:[%s] ", step.getTitle(), step.getExpected(), practice);
                    }
                }else {
                    throw new RuntimeException("不匹配的判断类型：" + step.getAssertion());
                }
            } catch (Exception e) {
                e.printStackTrace();
                success = false;
                message = String.format("error detected:[%s]", e.getMessage());
            }
            measures.add(new Measure(
                    step.getOrder(),
                    step.getTitle(),
                    success,
                    step.getAssertion(),
                    step.getExpected(),
                    practice,
                    message
            ));
        }
        return new Report(
                testCase.getUrl(),
                testCase.getBrowser(),
                measures
        );
    }


    private WebElement getWebElement(String getMethod, String key, WebDriver driver) {
        WebElement webElement;
        if (Objects.equals(getMethod, XPATH)) {
            webElement = driver.findElement(By.xpath(key));
        } else if (Objects.equals(getMethod, CSS)) {
            webElement = driver.findElement(By.cssSelector(key));
        } else if (Objects.equals(getMethod, ID)) {
            webElement = driver.findElement(By.id(key));
        } else if (Objects.equals(getMethod, NAME)) {
            webElement = driver.findElement(By.name(key));
        } else {
            throw new RuntimeException("未找到元素  type：" + getMethod + " key: " + key);
        }
        return webElement;
    }
}
