package com.tykj.datahouse.service;

import com.tykj.datahouse.entity.ColumnInfo;
import com.tykj.datahouse.entity.TableInfo;
import com.tykj.datahouse.repository.ColumnInfoRepository;
import com.tykj.datahouse.repository.TableInfoRepository;
import com.tykj.datahouse.sql.SqlCreator;
import com.tykj.datahouse.sql.impl.MysqlSqlCreator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;

/**
 * 表信息相关操作的Service
 * 涉及到对表的操作 均先进行对实际表的操作(执行sql语句) 再更新表信息
 */
@SuppressWarnings("Duplicates")
@Service
public class TableInfoService {

    @Autowired
    private TableInfoRepository tableInfoRepository;
    @Autowired
    private ColumnInfoRepository columnInfoRepository;
    @Autowired
    private DataSourceManager dataSourceManager;
    @Autowired
    private JdbcTemplate jdbcTemplate;

    private SqlCreator sqlCreator = new MysqlSqlCreator();

    /**
     * 创建新表
     *
     * @param tableInfo 要创建的表
     */
    public void createTable(TableInfo tableInfo) {
        dataSourceManager.clear();
        if (nonNull(tableInfo.getId())) {
            throw new RuntimeException("新增数据不可附带id");
        }
        long primaryCount = tableInfo.getColumnInfos().stream()
                .filter(ColumnInfo::getIsPrimary)
                .count();
        if (primaryCount != 1) {
            throw new RuntimeException("必须有且仅有1个主键");
        }

        dataSourceManager.switchToDataSource("target");
        String sql = sqlCreator.createTableSql(tableInfo);
        executeSQL(sql);

        dataSourceManager.clear();
        tableInfo.setCreatedTime(new Date());
        tableInfo.setUpdatedTime(new Date());
        TableInfo save = tableInfoRepository.save(tableInfo);
        Integer id = save.getId();
        if (nonNull(tableInfo.getColumnInfos())) {
            List<ColumnInfo> columnInfosForSave = tableInfo.getColumnInfos().stream()
                    .map(columnInfo -> columnInfo.setTableInfoId(id))
                    .collect(Collectors.toList());
            columnInfoRepository.saveAll(columnInfosForSave);
        }
    }

    /**
     * 总体思路:
     * 根据id查到更新之前的原表信息
     * 对比现表与原表的字段信息 筛选出新增字段、修改字段、删除字段
     * 要新增的字段进行新增、要修改的字段进行修改、要删除的字段进行删除
     * 更新完字段信息之后 最后再更新表信息
     */
    public void alterTable(TableInfo tableInfo) {
        //根据id查到更新之前的原表信息
        dataSourceManager.clear();
        if (isNull(tableInfo.getId())){
            throw new RuntimeException("修改数据必须附带id");
        }
        TableInfo oTableInfo = tableInfoRepository.findById(tableInfo.getId())
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", tableInfo.getId())));
        //对比现表与原表的字段信息 筛选出新增字段、修改字段、删除字段
        List<ColumnInfo> oColumnInfos = oTableInfo.getColumnInfos();
        List<ColumnInfo> columnInfos = tableInfo.getColumnInfos();
        Set<Integer> oColumnInfoIds = oColumnInfos.stream()
                .map(ColumnInfo::getId)
                .collect(Collectors.toSet());
        Set<Integer> columnInfoIds = columnInfos.stream()
                .map(ColumnInfo::getId)
                .collect(Collectors.toSet());
        Map<Integer, ColumnInfo> columnInfoMapForAlter = columnInfos.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(ColumnInfo::getId, columnInfo -> columnInfo));
        List<ColumnInfo> columnsForAdd = columnInfos.stream()
                .filter(columnInfo -> isForAdd(columnInfo.getId(), oColumnInfoIds))
                .collect(Collectors.toList());
        List<ColumnInfo> columnsForAlter = oColumnInfos.stream()
                .filter(oColumnInfo -> isForAlter(oColumnInfo.getId(), columnInfoIds))
                .collect(Collectors.toList());
        List<ColumnInfo> columnsForDelete = oColumnInfos.stream()
                .filter(oColumnInfo -> isForDelete(oColumnInfo.getId(), columnInfoIds))
                .collect(Collectors.toList());
        //要新增的字段进行新增、要修改的字段进行修改、要删除的字段进行删除
        for (ColumnInfo columnInfo : columnsForAdd) {
            dataSourceManager.switchToDataSource("target");
            String addColumnSql = sqlCreator.addColumnSql(tableInfo, columnInfo);
            jdbcTemplate.execute(addColumnSql);
            dataSourceManager.clear();
            columnInfoRepository.save(columnInfo.setTableInfoId(tableInfo.getId()));
        }
        for (ColumnInfo oColumnInfo : columnsForAlter) {
            dataSourceManager.switchToDataSource("target");
            String alterColumnSql = sqlCreator.alterColumnSql(tableInfo, oColumnInfo, columnInfoMapForAlter.get(oColumnInfo.getId()));
            jdbcTemplate.execute(alterColumnSql);
            dataSourceManager.clear();
            columnInfoRepository.save(columnInfoMapForAlter.get(oColumnInfo.getId()));
        }
        for (ColumnInfo oColumnInfo : columnsForDelete) {
            dataSourceManager.switchToDataSource("target");
            String removeColumnSql = sqlCreator.removeColumnSql(tableInfo, oColumnInfo);
            jdbcTemplate.execute(removeColumnSql);
            dataSourceManager.clear();
            columnInfoRepository.delete(oColumnInfo);
        }
        //更新完字段信息之后 最后再更新表信息
        dataSourceManager.clear();
        tableInfo.setCreatedTime(oTableInfo.getCreatedTime());
        tableInfo.setUpdatedTime(new Date());
        tableInfoRepository.save(oTableInfo);
    }

    /**
     * 根据id删除已存在的表
     *
     * @param id 指定删除的id
     */
    public void deleteTable(Integer id) {
        dataSourceManager.clear();
        TableInfo tableInfo = tableInfoRepository.findById(id)
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", id)));
        dataSourceManager.switchToDataSource("target");
        String dropTableSql = sqlCreator.dropTableSql(tableInfo);
        executeSQL(dropTableSql);
        dataSourceManager.clear();
        tableInfoRepository.deleteById(id);
        columnInfoRepository.deleteAllByTableInfoId(id);
    }

    public List<TableInfo> findAll() {
        return tableInfoRepository.findAll().stream()
                .map(this::getColumns)
                .collect(Collectors.toList());
    }

    public TableInfo findById(Integer id) {
        return tableInfoRepository.findById(id)
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", id)));
    }

    public void deleteAll() {
        tableInfoRepository.findAll().stream()
                .map(TableInfo::getId)
                .forEach(this::deleteTable);
        columnInfoRepository.deleteAll();
    }

    /**
     * 在指定表中插入一条数据
     *
     * @param tableId 指定表的id
     * @param row     Map形式的数据对象
     */
    public void insertData(Integer tableId, Map<String, Object> row) {
        dataSourceManager.clear();
        TableInfo tableInfo = tableInfoRepository.findById(tableId)
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", tableId)));
        dataSourceManager.switchToDataSource("target");
        String updateSql = sqlCreator.updateSql(tableInfo, row);
        jdbcTemplate.execute(updateSql);
        dataSourceManager.clear();
    }

    /**
     * 在指定表中删除一条数据
     *
     * @param tableId 指定表的id
     * @param row     Map形式的数据对象
     */
    public void deleteData(Integer tableId, Map<String, Object> row) {
        dataSourceManager.clear();
        TableInfo tableInfo = tableInfoRepository.findById(tableId)
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", tableId)));
        dataSourceManager.switchToDataSource("target");
        String deleteSql = sqlCreator.deleteSql(tableInfo, row);
        jdbcTemplate.execute(deleteSql);
        dataSourceManager.clear();
    }

    /**
     * 在指定表中更新一条数据
     *
     * @param tableId 指定表的id
     * @param row     Map形式的数据对象
     */
    public void updateData(Integer tableId, Map<String, Object> row) {
        dataSourceManager.clear();
        TableInfo tableInfo = tableInfoRepository.findById(tableId)
                .map(this::getColumns)
                .orElseThrow(() -> new RuntimeException(format("id为%s的表不存在", tableId)));
        dataSourceManager.switchToDataSource("target");
        String updateSql = sqlCreator.updateSql(tableInfo, row);
        jdbcTemplate.execute(updateSql);
        dataSourceManager.clear();
    }

    //---------------------------------------------------
    private Boolean isForAdd(Integer columnInfoId, Set<Integer> oColumnInfoIds) {
        return !oColumnInfoIds.contains(columnInfoId);
    }

    private Boolean isForAlter(Integer oColumnInfoId, Set<Integer> columnInfoIds) {
        return columnInfoIds.contains(oColumnInfoId);
    }

    private Boolean isForDelete(Integer oColumnInfoId, Set<Integer> columnInfoIds) {
        return !columnInfoIds.contains(oColumnInfoId);
    }

    private TableInfo getColumns(TableInfo tableInfo) {
        List<ColumnInfo> columnInfos = columnInfoRepository.findByTableInfoId(tableInfo.getId());
        tableInfo.setColumnInfos(columnInfos);
        return tableInfo;
    }

    private void executeSQL(String sql) {
        System.out.println(sql);
        jdbcTemplate.execute(sql);
    }
}
