package com.tykj.datahouse.sql.impl;

import com.tykj.datahouse.entity.ColumnInfo;
import com.tykj.datahouse.entity.TableInfo;
import com.tykj.datahouse.sql.SqlCreator;

import java.util.List;
import java.util.Map;

import static java.lang.String.format;
import static java.util.Objects.nonNull;

public class MysqlSqlCreator implements SqlCreator {

    @Override
    public String createTableSql(TableInfo tableInfo) {
        List<ColumnInfo> columnInfos = tableInfo.getColumnInfos();
        ColumnInfo primary = columnInfos.stream()
                .filter(ColumnInfo::getIsPrimary)
                .findAny()
                .orElseThrow(() -> new RuntimeException("没有主键"));
        StringBuilder result = new StringBuilder();
        //起始
        result.append(format("CREATE TABLE `%s`(\n", tableInfo.getName()));
        //其他字段
        for (ColumnInfo columnInfo : columnInfos) {
            result.append(format("`%s` %s", columnInfo.getName(), columnInfo.getType()));
            if (nonNull(columnInfo.getLength())) {
                result.append(format("(%s)", columnInfo.getLength()));
            }
            if (nonNull(columnInfo.getNotNull())) {
                result.append(" not null");
            }
            result.append(",\n");
        }
        //主键
        result.append(format("PRIMARY KEY (%s)\n", primary.getName()));
        //结尾
        result.append(")");
        return result.toString();
    }

    @Override
    public String addColumnSql(TableInfo tableInfo, ColumnInfo columnInfo) {
        StringBuilder result = new StringBuilder();
        result.append(format("ALTER TABLE %s ADD COLUMN %s %s", tableInfo.getName(), columnInfo.getName(), columnInfo.getType()));
        if (nonNull(columnInfo.getLength())) {
            result.append(format("(%s)", columnInfo.getLength()));
        }
        if (nonNull(columnInfo.getNotNull())) {
            result.append(" not null");
        }
        return result.toString();
    }

    @Override
    public String alterColumnSql(TableInfo tableInfo, ColumnInfo oColumnInfo, ColumnInfo columnInfo) {
        StringBuilder result = new StringBuilder();
        result.append(format("ALTER TABLE `%s` CHANGE COLUMN `%s` `%s` %s", tableInfo.getName(), oColumnInfo.getName(), columnInfo.getName(), columnInfo.getType()));
        if (nonNull(columnInfo.getLength())) {
            result.append(format("(%s)", columnInfo.getLength()));
        }
        if (nonNull(columnInfo.getNotNull())) {
            result.append(" not null");
        }
        return result.toString();
    }

    @Override
    public String removeColumnSql(TableInfo tableInfo, ColumnInfo columnInfo) {
        return format("ALTER TABLE `%s` DROP COLUMN `%s`", tableInfo.getName(), columnInfo.getName());
    }

    @Override
    public String dropTableSql(TableInfo tableInfo) {
        return format("DROP TABLE `%s`", tableInfo.getName());
    }

    @Override
    public String insertSql(TableInfo tableInfo, Map<String, Object> row) {
        List<ColumnInfo> columnInfos = tableInfo.getColumnInfos();
        StringBuilder result = new StringBuilder();
        StringBuilder columns = new StringBuilder();
        StringBuilder values = new StringBuilder();
        //起始
        result.append(format("INSERT INTO `%s` ", tableInfo.getName()));
        for (ColumnInfo columnInfo : columnInfos) {
            columns.append(format("`%s`", columnInfo.getName())).append(",");
            values.append(format("\'%s\',", row.get(columnInfo.getName())));
        }
        int columnLastIndex = columns.lastIndexOf(",");
        if (columnLastIndex > 0) {
            columns.delete(columnLastIndex, columns.length());
        }
        int valueLastIndex = values.lastIndexOf(",");
        if (valueLastIndex > 0) {
            values.delete(valueLastIndex, values.length());
        }
        result.append(format("(%s) VALUES (%s)", columns.toString(), values.toString()));
        return result.toString();
    }

    @Override
    public String updateSql(TableInfo tableInfo, Map<String, Object> row) {
        List<ColumnInfo> columnInfos = tableInfo.getColumnInfos();
        ColumnInfo primary = columnInfos.stream()
                .filter(ColumnInfo::getIsPrimary)
                .findAny()
                .orElseThrow(() -> new RuntimeException("没有主键"));
        columnInfos.remove(primary);
        StringBuilder result = new StringBuilder();
        result.append(format("UPDATE `%s` SET ", tableInfo.getName()));
        for (ColumnInfo columnInfo : columnInfos) {
            result.append(format("`%s` = \'%s\',", columnInfo.getName(), row.get(columnInfo.getName())));
        }
        int lastIndex = result.lastIndexOf(",");
        if (lastIndex > 0) {
            result.delete(lastIndex, result.length());
        }
        result.append(format(" WHERE (`%s` = \'%s\')", primary.getName(), row.get(primary.getName())));
        return result.toString();
    }

    @Override
    public String deleteSql(TableInfo tableInfo, Map<String, Object> row) {
        StringBuilder result = new StringBuilder();
        ColumnInfo primary = tableInfo.getColumnInfos().stream()
                .filter(ColumnInfo::getIsPrimary)
                .findAny()
                .orElseThrow(() -> new RuntimeException("没有主键"));
        result.append(format("DELETE FROM `%s` WHERE (`%s` = \'%s\')", tableInfo.getName(), primary.getName(), row.get(primary.getName())));
        return result.toString();
    }

    @Override
    public String querySql(String tableName, List<String> columns) {
        StringBuilder result = new StringBuilder();

        return result.toString();
    }

}
