package com.example.sqlserver.collector.service;

import com.example.sqlserver.collector.entity.local.Article;
import com.example.sqlserver.collector.entity.local.Annex;
import com.example.sqlserver.collector.entity.local.Channel;
import com.example.sqlserver.collector.entity.remote.RemoteArticle;
import com.example.sqlserver.collector.entity.remote.RemoteArticleInfo;
import com.example.sqlserver.collector.entity.remote.RemoteChannel;
import com.example.sqlserver.collector.repository.local.ArticleRepository;
import com.example.sqlserver.collector.repository.local.AnnexRepository;
import com.example.sqlserver.collector.repository.local.ChannelRepository;
import com.example.sqlserver.collector.repository.remote.RemoteArticleInfoRepository;
import com.example.sqlserver.collector.repository.remote.RemoteArticleRepository;
import com.example.sqlserver.collector.repository.remote.RemoteChannelRepository;
import com.example.sqlserver.collector.service.helper.ContentParseHelper;
import com.example.sqlserver.collector.service.helper.TransHelper;
import com.example.sqlserver.collector.utils.OssUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>Description : sqlserver-test
 * <p>Date : 2019/7/26 14:08
 * <p>@author : C
 */
@SuppressWarnings("SpringAutowiredFieldsWarningInspection")
@Slf4j
@Service
public class Updater {
    @Autowired
    RemoteArticleRepository remoteArticleRepository;
    @Autowired
    RemoteArticleInfoRepository remoteArticleInfoRepository;
    @Autowired
    RemoteChannelRepository remoteChannelRepository;
    @Autowired
    ArticleRepository articleRepository;
    @Autowired
    AnnexRepository annexRepository;
    @Autowired
    ChannelRepository channelRepository;
    @Autowired
    TransHelper transHelper;
    @Autowired
    ContentParseHelper contentParseHelper;
    @Autowired
    OssUtil ossUtil;
    @Value("${oss.bucket}")
    String bucket;
    /**
     * 待更新文章队列
     */
    private List<RemoteArticle> remoteArticleList = Lists.newArrayList();

    /**
     * 待更新栏目队列
     */
    private List<RemoteChannel> remoteChannelList = Lists.newArrayList();

    /**
     * 待更新附件路径队列
     */
    private List<String> pathList = Lists.newArrayList();


    private Map<Channel, List<Channel>> channelSubMap = Maps.newHashMap();

    Map<Channel, List<Channel>> channelArticleMap = Maps.newHashMap();

    private Integer channelCount = 0;

    private Integer articleCount = 0;

    private Integer annexCount = 0;

    private Set<String> savedChannelIds = Sets.newHashSet();

    public void updateAll() {
        log.info("开始采集数据...");
        log.info("[栏目信息]待更新数据总条数 : {}", remoteChannelRepository.count());
        log.info("[文章信息]待更新数据总条数 : {}", remoteArticleInfoRepository.count());
        remoteChannelList = remoteChannelRepository.findAll();
        remoteChannelList.forEach(this::updateHandle);
    }

    public void updateHandle(RemoteChannel remoteChannel) {
        String id = remoteChannel.getId();
        Channel keyChannel = transHelper.toChannel(remoteChannel);
        if (isNew(remoteChannel)){
            List<Channel> channelsForSave = remoteChannelList.stream()
                    .filter(remoteChannel1 -> isSub(remoteChannel1, id))
                    .map(transHelper::toChannel)
                    .map(channel -> channel.replaceSuperUUID(keyChannel.getUuid()))
                    .collect(Collectors.toList());
            savedChannelIds.add(id);
            savedChannelIds.addAll(remoteChannelList.stream()
                    .filter(remoteChannel1 -> isSub(remoteChannel1, id))
                    .map(RemoteChannel::getId)
                    .collect(Collectors.toList())
            );
            save(keyChannel);
            saveChannels(channelsForSave);
            channelCount++;
            channelCount += channelsForSave.size();
            log.info("[栏目信息]已保存 : {}", channelCount);
        }

        List<Article> articlesForSave = remoteArticleInfoRepository.findAllByChId(id).stream()
                .map(transHelper::toArticle)
                .map(article -> article.replaceChUUID(keyChannel.getUuid()))
                .map(transHelper::replaceContent)
                .collect(Collectors.toList());
        saveArticles(articlesForSave);
        articleCount += articlesForSave.size();
        log.info("[文章信息]已保存 : {}", articleCount);
        List<Annex> annexesForSave = articlesForSave.stream()
                .flatMap(transHelper::toAnnexStream)
                .filter(contentParseHelper::isFile)
                .collect(Collectors.toList());
        saveAnnexes(annexesForSave);
        annexCount += annexesForSave.size();
        log.info("[附件信息]已保存 : {}", annexCount);

    }

    private Boolean isNew(RemoteChannel remoteChannel) {
        return savedChannelIds.stream().noneMatch(id -> Objects.equals(remoteChannel.getId(), id));
    }

    private Boolean isSub(RemoteChannel remoteChannel, String id) {
        return remoteChannel.getId().length() > id.length() && remoteChannel.getId().startsWith(id);
    }

    private Stream<RemoteArticleInfo> getArticleInfoStream(String id) {
        return remoteArticleInfoRepository.findAllByChId(id).stream();
    }

    public void getRemoteArticlesForSave() {
        if (remoteArticleList.isEmpty()) {
            List<RemoteArticle> remoteArticlesForSave = remoteArticleRepository.findAll();
            remoteArticleList.addAll(remoteArticlesForSave);
            log.info("获取文章队列完毕 需要更新数量剩余 : {} ", remoteArticleList.size());
        }
    }

    public void getRemoteChannelsForSave() {
        if (remoteChannelList.isEmpty()) {
            List<RemoteChannel> remoteChannelsForSave = remoteChannelRepository.findAll();
            remoteChannelList.addAll(remoteChannelsForSave);
            log.info("获取栏目队列完毕 需要更新数量剩余 : {} ", remoteChannelsForSave.size());
        }
    }

    /**
     * @param n 一次从待更新集合中取多少条
     */
    public void updateChannels(Integer n) {
        List<RemoteChannel> remoteChannelListForSave = remoteChannelList.stream()
                .limit(n)
                .collect(Collectors.toList());
        remoteChannelList.removeAll(remoteChannelListForSave);
        List<Channel> channelList = remoteChannelListForSave.stream()
                .map(transHelper::toChannel)
                .collect(Collectors.toList());
        channelList.forEach(this::save);
        if (!remoteChannelList.isEmpty()) {
            log.info("栏目待更新数量 : {} ", remoteChannelList.size());
        }
    }

    /**
     * @param n 一次从待更新集合中取多少条
     */
    public void updateArticlesAndAnnexes(Integer n) {
        List<RemoteArticle> remoteArticleListForSave = remoteArticleList.stream()
                .limit(n)
                .collect(Collectors.toList());
        List<Article> articleList = remoteArticleListForSave.stream()
                .filter(transHelper::infoExists)
                .map(transHelper::toArticle)
                .map(transHelper::replaceContent)
                .map(articleRepository::save)
                .collect(Collectors.toList());
        remoteArticleList.removeAll(remoteArticleListForSave);
        List<Annex> annexList = articleList.stream()
                .flatMap(transHelper::toAnnexStream)
                .filter(contentParseHelper::isFile)
                .collect(Collectors.toList());
        List<String> pathsForUpload = articleList.stream()
                .flatMap(transHelper::toPathStream)
                .map(contentParseHelper::restorePath)
                .collect(Collectors.toList());
        pathList.addAll(pathsForUpload);
        if (!annexList.isEmpty()) {
            log.info("本次文章更新带有附件数量 : {}", annexList.size());
        }
        annexList.forEach(this::save);
        if (!remoteArticleList.isEmpty()) {
            log.info("文章待更新数量 : {}", remoteArticleList.size());
        }
    }

    /**
     * @param n 一次从待更新集合中取多少条
     */
    public void update(Integer n) {
        List<RemoteChannel> remoteChannelListForSave = remoteChannelList.stream()
                .limit(n)
                .collect(Collectors.toList());
        List<Channel> channelList = remoteChannelListForSave.stream()
                .map(transHelper::toChannel)
                .map(channelRepository::save)
                .collect(Collectors.toList());
        remoteChannelList.removeAll(remoteChannelListForSave);
        if (!remoteChannelList.isEmpty()) {
            log.info("栏目待更新数量 : {} ", remoteChannelList.size());
        }
        List<RemoteArticle> remoteArticleListForSave = new ArrayList<>(remoteArticleList);
        List<Article> articleList = remoteArticleListForSave.stream()
                .filter(transHelper::infoExists)
                .map(transHelper::toArticle)
                .map(transHelper::replaceContent)
                .map(articleRepository::save)
                .collect(Collectors.toList());
        remoteArticleList.removeAll(remoteArticleListForSave);
        List<Annex> annexList = articleList.stream()
                .flatMap(transHelper::toAnnexStream)
                .filter(contentParseHelper::isFile)
                .collect(Collectors.toList());
        List<String> pathsForUpload = articleList.stream()
                .flatMap(transHelper::toPathStream)
                .map(contentParseHelper::restorePath)
                .collect(Collectors.toList());
        pathList.addAll(pathsForUpload);
        if (!annexList.isEmpty()) {
            log.info("本次文章更新带有附件数量 : {}", annexList.size());
        }
        annexList.forEach(this::save);
        if (!remoteArticleList.isEmpty()) {
            log.info("文章待更新数量 : {}", remoteArticleList.size());
        }
    }

    /**
     * @param n 一次从待更新集合中取多少条
     */
    public void uploadFile(Integer n) {
        List<String> filesForUpload = pathList.stream()
                .limit(n)
                .collect(Collectors.toList());
        filesForUpload.forEach(this::upload);
    }

    private void save(Channel channel) {
        channelRepository.save(channel);
    }

    private void saveChannels(List<Channel> channels) {
        channelRepository.save(channels);
    }

    private void save(Article article) {
        articleRepository.save(article);
    }

    private void saveArticles(List<Article> articles) {
        articleRepository.save(articles);
    }

    private void save(Annex annex) {
        annexRepository.save(annex);
    }

    private void saveAnnexes(List<Annex> annexes) {
        annexRepository.save(annexes);
    }

    private void upload(String path) {
        try {
            InputStream inputStream = Files.newInputStream(Paths.get(System.getProperty("user.dir") + path), StandardOpenOption.READ);
            ossUtil.uploadFile(bucket, transHelper.getFileName(path), inputStream);
            log.info("文件上传完毕    路径 : {} ", path);
        } catch (IOException e) {
            log.error("文件上传失败  路径 : {}  异常 : {}", path, e);
        }
    }

}
