package com.example.morsecoder;

import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * 摩斯码转换器
 * 摩尔斯电码由三种类型的信号组成,分别为:短信号(滴)、长信号(嗒)和分隔符。三种信号通常习惯使用“.”、“-”、“/”表示。
 *
 *
 * 标准字典
 * 注意：字母都会转换为大写，0 为短信号，1 为长信号。
 *
 * 字符	电码
 * A	01
 * B	1000
 * C	1010
 * D	100
 * E	0
 * F	0010
 * G	110
 * H	0000
 * I	00
 * J	0111
 * K	101
 * L	0100
 * M	11
 * N	10
 * O	111
 * P	0110
 * Q	1101
 * R	010
 * S	000
 * T	1
 * U	001
 * V	0001
 * W	011
 * X	1001
 * Y	1011
 * Z	1100
 *
 * 字符	电码
 * 0	11111
 * 1	01111
 * 2	00111
 * 3	00011
 * 4	00001
 * 5	00000
 * 6	10000
 * 7	11000
 * 8	11100
 * 9	11110
 *
 * 字符	电码
 * .	010101
 * ,	110011
 * ?	001100
 * '	011110
 * !	101011
 * /	10010
 * (	10110
 * )	101101
 * &	01000
 * :	111000
 * ;	101010
 * =	10001
 * +	01010
 * -	100001
 * _	001101
 * "	010010
 * $	0001001
 * @	011010
 *
 * 其他 Unicode 字符
 * 编码时，直接将 codePoint 转换为 2 进制字符串，然后 0 替换为短信号，1 替换为长信号。
 *
 * 解码时反之。
 *
 */
@Service
public class MorseCoder {

    private static final Map<Integer, String> alphabets = new HashMap<>();    // code point -> morse
    private static final Map<String, Integer> dictionaries = new HashMap<>(); // morse -> code point

    private static void registerMorse(Character abc, String dict) {
        alphabets.put(Integer.valueOf(abc), dict);
        dictionaries.put(dict, Integer.valueOf(abc));
    }

    static {
        // Letters
        registerMorse('A', "01");
        registerMorse('B', "1000");
        registerMorse('C', "1010");
        registerMorse('D', "100");
        registerMorse('E', "0");
        registerMorse('F', "0010");
        registerMorse('G', "110");
        registerMorse('H', "0000");
        registerMorse('I', "00");
        registerMorse('J', "0111");
        registerMorse('K', "101");
        registerMorse('L', "0100");
        registerMorse('M', "11");
        registerMorse('N', "10");
        registerMorse('O', "111");
        registerMorse('P', "0110");
        registerMorse('Q', "1101");
        registerMorse('R', "010");
        registerMorse('S', "000");
        registerMorse('T', "1");
        registerMorse('U', "001");
        registerMorse('V', "0001");
        registerMorse('W', "011");
        registerMorse('X', "1001");
        registerMorse('Y', "1011");
        registerMorse('Z', "1100");
        // Numbers
        registerMorse('0', "11111");
        registerMorse('1', "01111");
        registerMorse('2', "00111");
        registerMorse('3', "00011");
        registerMorse('4', "00001");
        registerMorse('5', "00000");
        registerMorse('6', "10000");
        registerMorse('7', "11000");
        registerMorse('8', "11100");
        registerMorse('9', "11110");
        // Punctuation
        registerMorse('.', "010101");
        registerMorse(',', "110011");
        registerMorse('?', "001100");
        registerMorse('\'', "011110");
        registerMorse('!', "101011");
        registerMorse('/', "10010");
        registerMorse('(', "10110");
        registerMorse(')', "101101");
        registerMorse('&', "01000");
        registerMorse(':', "111000");
        registerMorse(';', "101010");
        registerMorse('=', "10001");
        registerMorse('+', "01010");
        registerMorse('-', "100001");
        registerMorse('_', "001101");
        registerMorse('"', "010010");
        registerMorse('$', "0001001");
        registerMorse('@', "011010");
    }

    private final char dit; // short mark or dot
    private final char dah; // longer mark or dash
    private final char split;

    public MorseCoder() {
        this('.', '-', '/');
    }

    public MorseCoder(char dit, char dah, char split) {
        this.dit = dit;
        this.dah = dah;
        this.split = split;
    }

    public String encode(String text) {
        if (text == null) {
            throw new IllegalArgumentException("Text should not be null.");
        }
        StringBuilder morseBuilder = new StringBuilder();
        text = text.toUpperCase();
        for (int i = 0; i < text.codePointCount(0, text.length()); i++) {
            int codePoint = text.codePointAt(text.offsetByCodePoints(0, i));
            String word = alphabets.get(codePoint);
            if (word == null) {
                word = Integer.toBinaryString(codePoint);
            }
            morseBuilder.append(word.replace('0', dit).replace('1', dah)).append(split);
        }
        return morseBuilder.toString();
    }

    public String decode(String morse) {
        if (morse == null) {
            throw new IllegalArgumentException("Morse should not be null.");
        }
        StringBuilder textBuilder = new StringBuilder();
        StringTokenizer tokenizer = new StringTokenizer(morse, String.valueOf(split));
        while (tokenizer.hasMoreTokens()) {
            String word = tokenizer.nextToken().replace(dit, '0').replace(dah, '1');
            Integer codePoint = dictionaries.get(word);
            if (codePoint == null) {
                codePoint = Integer.valueOf(word, 2);
            }
            textBuilder.appendCodePoint(codePoint);
        }
        return textBuilder.toString();
    }


    //HELLO WORLD!
    public static void main(String[] args) {
        MorseCoder morseCoder = new MorseCoder();
        String decode = morseCoder.decode("-..----.--...../-.--..-.-----.-/");
        System.out.println(decode);
        String encode = morseCoder.encode("你好");
        System.out.println(encode);
    }

}
