package com.example.datasending;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.squareup.okhttp.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

@RestController
@SpringBootApplication
@RequestMapping("/simc")
public class ApplicationController {

    public static void main(String[] args) {
        SpringApplication.run(ApplicationController.class, args);
    }

    private static final Logger log = LoggerFactory.getLogger(ApplicationController.class);
    private static final String IP = "http://localhost:82/schedule/terminal";
    private static final MediaType JSON = MediaType.parse("application/json; charset=utf-8");
    private static final int THREAD_POOL_SIZE = 2;
    private final OkHttpClient httpClient = new OkHttpClient();
    private ThreadPoolTaskExecutor taskExecutor;
    private Alamofire alamofire;
    private ObjectMapper objectMapper;
    private Map<String, Boolean> tasks;

    public ApplicationController() {
        taskExecutor = new ThreadPoolTaskExecutor();
        taskExecutor.setCorePoolSize(THREAD_POOL_SIZE);
        taskExecutor.setMaxPoolSize(THREAD_POOL_SIZE);
        taskExecutor.setQueueCapacity(1);
        taskExecutor.initialize();
        objectMapper = new ObjectMapper();
        tasks = new ConcurrentHashMap<>();
        alamofire = new Alamofire();
    }

    @GetMapping("/start/{taskName}/{interval}")
    public ResponseEntity<String> startTask(@PathVariable String taskName, @PathVariable long interval) {
        if (tasks.containsKey(taskName)) {
            return ResponseEntity.badRequest().body("该任务已在运行。");
        }
        tasks.put(taskName, true);

        Runnable task;
        switch (taskName) {
            case "task1":
                task = () -> sendTerminalDataTask(tasks, taskName, () -> new TerminalData().initTerminalData(),interval);
                break;
            case "task2":
                task = () -> sendTerminalDataTask(tasks, taskName, () -> new TerminalData().initTerminalData1(),interval);
                break;
            default:
                tasks.remove(taskName);
                return ResponseEntity.badRequest().body("无效的任务名称。");
        }
        taskExecutor.submit(task);
        return ResponseEntity.ok(taskName + "已启动。");
    }

    @GetMapping("/stop/{taskName}")
    public ResponseEntity<String> stopTask(@PathVariable String taskName) {
        if (!tasks.containsKey(taskName)) {
            return ResponseEntity.badRequest().body("任务不存在。");
        }
        tasks.put(taskName, false);
        return ResponseEntity.ok(taskName + "已停止。");
    }

    @GetMapping("/tasks")
    public ResponseEntity<String> getRunningTasks() {
        StringBuilder tasksStatus = new StringBuilder();
        tasks.forEach((taskName, isRunning) -> tasksStatus
                .append(taskName)
                .append("：")
                .append(isRunning ? "正在运行" : "已停止")
                .append("\n")
        );
        return ResponseEntity.ok(tasksStatus.toString());
    }

    private void sendTerminalDataTask(Map<String, Boolean> tasks, String taskName, Supplier<TerminalData> terminalDataSupplier, long interval) {
        while (tasks.get(taskName)) {
            TerminalData terminalData = terminalDataSupplier.get();
            String json = null;
            try {
                json = objectMapper.writeValueAsString(terminalData);
            } catch (JsonProcessingException e) {
                e.printStackTrace();
            }
            try {
                alamofire.POST(IP, json, httpClient);
                Thread.sleep(interval); // 添加间隔
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private static class Alamofire {
        private String POST(final String url, final String json, OkHttpClient httpClient) throws IOException {
            RequestBody body = RequestBody.create(JSON, json);
            Request request = new Request.Builder()
                    .url(url)
                    .post(body)
                    .build();
            Response response = httpClient.newCall(request).execute();
            return response.body().string();
        }
    }
}
