package com.tykj.dev.device.destroy.controller;

import com.fasterxml.jackson.core.type.TypeReference;
import com.tykj.dev.config.base.DeviceLifeStatus;
import com.tykj.dev.config.swagger.AutoDocument;
import com.tykj.dev.device.destroy.entity.domain.DeviceDestroyBill;
import com.tykj.dev.device.destroy.entity.enums.DestroyStatus;
import com.tykj.dev.device.destroy.entity.vo.*;
import com.tykj.dev.device.destroy.service.DeviceDestroyBillService;
import com.tykj.dev.device.library.repository.DeviceLibraryDao;
import com.tykj.dev.device.library.service.DeviceLibraryService;
import com.tykj.dev.device.library.service.DeviceLogService;
import com.tykj.dev.device.library.subject.Dto.DeviceLogDto;
import com.tykj.dev.device.library.subject.domin.DeviceLibrary;
import com.tykj.dev.device.library.subject.vo.DeviceLibrarySelectVo;
import com.tykj.dev.device.library.subject.vo.FileVo;
import com.tykj.dev.device.library.subject.vo.ScriptSaveVo;
import com.tykj.dev.device.task.service.TaskLogService;
import com.tykj.dev.device.task.service.TaskService;
import com.tykj.dev.device.task.subject.bto.TaskBto;
import com.tykj.dev.device.user.read.service.MessageService;
import com.tykj.dev.device.user.read.subject.bto.MessageBto;
import com.tykj.dev.device.user.subject.entity.User;
import com.tykj.dev.device.user.subject.service.UserPublicService;
import com.tykj.dev.device.user.util.AuthenticationUtils;
import com.tykj.dev.device.user.util.UserUtils;
import com.tykj.dev.misc.base.BusinessEnum;
import com.tykj.dev.misc.base.StatusEnum;
import com.tykj.dev.misc.utils.*;
import com.tykj.dev.rfid.service.InputOutputDeviceService;
import com.tykj.dev.socket.MyWebSocket;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotNull;
import java.text.ParseException;
import java.util.*;

/**
 * @author huangxiahao
 */
@RestController
@RequestMapping(value = "/destroy")
@AutoDocument
@Api(tags = "销毁业务模块", value = "销毁业务模块")
public class DeviceDestroyController {

    /**
     * 装备销毁对应的生命周期编号
     */
    private final DeviceDestroyBillService deviceDestroyBillService;
    private final TaskService taskService;
    private final DeviceLibraryService deviceLibraryService;
    private final UserPublicService userPublicService;
    private final DeviceLogService deviceLogService;
    private final TaskLogService taskLogService;
    @Autowired
    MyWebSocket myWebSocket;
    @Autowired
    private AuthenticationUtils AuthenticationUtils;
    @Autowired
    private UserUtils userUtils;
    @Autowired
    private InputOutputDeviceService inputOutputDeviceService;
    @Autowired
    private DeviceLibraryDao deviceLibraryDao;
    @Autowired
    private MessageService messageService;

    public DeviceDestroyController(DeviceDestroyBillService deviceDestroyBillServiceyBillService, TaskService taskService, DeviceLibraryService deviceLibraryService, UserPublicService userPublicService, DeviceLogService deviceLogService, TaskLogService taskLogService) {
        this.deviceDestroyBillService = deviceDestroyBillServiceyBillService;
        this.taskService = taskService;
        this.deviceLibraryService = deviceLibraryService;
        this.userPublicService = userPublicService;
        this.deviceLogService = deviceLogService;
        this.taskLogService = taskLogService;
    }

    @ApiOperation(value = "查询销毁单", notes = "可以通过这个接口查询销毁单")
    @PostMapping(value = "/summary")
    public ResponseEntity selectRepairBill(@RequestBody DeviceDestroyBillSelectVo deviceDestoryBillSelectVo) {
        return ResultUtil.success(deviceDestroyBillService.getPage(deviceDestoryBillSelectVo, deviceDestoryBillSelectVo.getPageable()));
    }

    @ApiOperation(value = "提交销毁表单", notes = "可以通过这个接口提交销毁表单发起销毁流程")
    @PostMapping(value = "/form")
    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity selectRepairBill(@Validated @RequestBody DeviceDestroyFormVo formVo) throws ParseException {
        deviceLibraryService.isWaitRetired(formVo.getDevices());
        //生成销毁账单
        User currentUserInfo = AuthenticationUtils.getAuthentication().getCurrentUserInfo();
        DeviceDestroyBill destroyBill = DeviceDestroyBill.formVoToBill(formVo, currentUserInfo);
        //更改账单状态
        destroyBill.setDestroyStatus(DestroyStatus.CONFIRM_SUCCESS.getStatus());
        //设置销毁时间
        destroyBill.setDestroyTime(formVo.getDestroyTime());
        destroyBill.setScriptJson(JacksonUtil.toJSon(formVo.getScriptSaveVos()));
        //保存销毁账单
        DeviceDestroyBill deviceDestroyBillEntitySaved = deviceDestroyBillService.addEntity(destroyBill);
        //为账单新增文号
        deviceDestroyBillEntitySaved.setDocNumber(this.deviceDestroyBillService.getNewDocNumber());
        deviceDestroyBillService.updateEntity(deviceDestroyBillEntitySaved);
        //生成日志所需的文件列表
        ArrayList<FileVo> billList = ListUtil.createList(
                new FileVo("出库确认单", deviceDestroyBillEntitySaved.getFileName(), deviceDestroyBillEntitySaved.getFileUrl())
        );
        //生成装备变更日志
        formVo.getDevices().forEach(d -> {
            deviceLogService.addLog(new DeviceLogDto(d, "发起销毁", null));
        });
        //更改对应装备的状态为销毁
        handleDeviceDetail(deviceDestroyBillEntitySaved, null, 0);
        //生成上传销毁单任务
        TaskBto task = taskService.start(initTaskBto(deviceDestroyBillEntitySaved));
        //发送阅知信息
        List<Integer> userIds = userPublicService.findOtherUser(userUtils.getCurrentUserId());
        MessageBto messageBto = new MessageBto(task.getId(),task.getBusinessType(),"发起销毁",userIds,1);
        messageService.add(messageBto);
        myWebSocket.sendMessage1();
        return ResponseEntity.ok(selectDestroyDetail(deviceDestroyBillEntitySaved.getId()));
    }

    @ApiOperation(value = "查询可销毁装备列表", notes = "通过这个接口查询可销毁装备接口")
    @PostMapping(value = "/devices")
    public ResponseEntity selectDevices(@RequestBody CanDestroyDeviceSelectVo canDestoryDeviceSelectVo) {
        DeviceLibrarySelectVo deviceLibrarySelectVo = new DeviceLibrarySelectVo();
        BeanUtils.copyProperties(canDestoryDeviceSelectVo, deviceLibrarySelectVo);
        deviceLibrarySelectVo.setLifeStatus(Arrays.asList(2, 5, 12));
        List<DeviceLibrary> list = deviceLibraryService.getList(deviceLibrarySelectVo);
        return ResultUtil.success(list);
    }

    @ApiOperation(value = "查询销毁单详情", notes = "可以通过这个接口查询销毁单")
    @GetMapping(value = "/detail")
    public ResponseEntity selectDestroyDetail(@Validated @NotNull Integer destroyId) {
        Map<Integer,DeviceLibrary> deviceLibraryMap = deviceLibraryService.getAllDeviceMap();
        //装备文号
        String docNumber;
        DeviceDestroyDetailResultVo destroyDetailResultVo;
        Set<String> nameSet = new HashSet<>();
        Set<String> modelSet = new HashSet<>();
        List<DeviceLibrary> devices = new ArrayList<>();
        DeviceDestroyBill destroyBill = deviceDestroyBillService.getOne(destroyId);
        //获取销毁详情里的装备
        List<Integer> destroyDetail = StringSplitUtil.split(destroyBill.getDestroyDeviceDetail());
        for (Integer id : destroyDetail) {
            if (id > 0) {
                DeviceLibrary d = deviceLibraryMap.get(id);
                devices.add(d);
                nameSet.add(d.getName());
                modelSet.add(d.getModel());
            }
        }
        //如果装备已经被销毁则生成拼接文号
        docNumber = makeDocNumber(destroyBill);
        //拼接ResultVo
        destroyDetailResultVo = new DeviceDestroyDetailResultVo(
                destroyBill.getStartUserAId(),
                destroyBill.getStartUserBId(),
                userPublicService.getOne(destroyBill.getStartUserAId()).getName(),
                userPublicService.getOne(destroyBill.getStartUserBId()).getName(),
                destroyBill.getDestroyStatus(),
                destroyBill.getDestroyFileName(),
                destroyBill.getDestroyFileUrl(),
                docNumber,
                destroyBill.getDestroyTime(),
                destroyBill.getUndertaker(),
                destroyBill.getLeader(),
                destroyBill.getSupervisor(),
                StringUtils.join(nameSet, ","),
                StringUtils.join(modelSet, ","),
                userPublicService.findUnitsNameByUserId(destroyBill.getStartUserAId()),
                devices,
                JacksonUtil.readValue(destroyBill.getScriptJson(), new TypeReference<List<ScriptSaveVo>>() {
                })

        );
        return ResultUtil.success(destroyDetailResultVo);
    }

    @ApiOperation(value = "销毁出库单确认", notes = "可以通过这个确认销毁单")
    @PutMapping(value = "/confirmation")
    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity confirmDestroy(@RequestBody @Validated DeviceDestroyConfirmVo deviceDestoryConfirmVo) throws ParseException {
        //初始化需要的数据
        TaskBto task = taskService.get(deviceDestoryConfirmVo.getTaskId());
        DeviceDestroyBill deviceDestroyBill = deviceDestroyBillService.getOne(task.getBillId());
        //判断审核结果 0为审核通过，1为审核不通过
        if (deviceDestoryConfirmVo.getResult() == 1) {
            //更改账单状态
            deviceDestroyBill.setDestroyStatus(DestroyStatus.CONFIRM_FAILED.getStatus());
            //更新销毁账单
            this.deviceDestroyBillService.updateEntity(deviceDestroyBill);
            //将任务推进至封存
            taskService.moveToArchive(task);
        } else {
            //更改账单状态
            deviceDestroyBill.setDestroyStatus(DestroyStatus.CONFIRM_SUCCESS.getStatus());
            //设置销毁时间
            deviceDestroyBill.setDestroyTime(new Date());
            //为账单新增文号
            deviceDestroyBill.setDocNumber(this.deviceDestroyBillService.getNewDocNumber());
            //更新销毁账单
            this.deviceDestroyBillService.updateEntity(deviceDestroyBill);
            //将任务推进至下一步
            taskService.moveToNext(task, deviceDestroyBill.getStartUserBId());
            //更改对应装备的状态为销毁
            handleDeviceDetail(deviceDestroyBill, null, deviceDestoryConfirmVo.getResult());
        }
        myWebSocket.sendMessage1();
        return selectDestroyDetail(deviceDestroyBill.getId());
    }

    @ApiOperation(value = "上传销毁单", notes = "可以通过这个上传销毁单")
    @PostMapping(value = "/upload")
    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity uploadDestroyFile(@RequestBody @Validated UploadDestroyFileVo uploadDestroyFileVo) {
        //获取当前task
        TaskBto taskBto = taskService.get(uploadDestroyFileVo.getTaskId());
        TaskDisposeUtil.isNotSubmit(taskBto.getBillStatus(),StatusEnum.WAIT_UPLOAD_BILL_FILE);
        //获取销毁单
        DeviceDestroyBill deviceDestroyBill = deviceDestroyBillService.getOne(taskBto.getBillId());
        //更新销毁单
        if (uploadDestroyFileVo.getDestroyFileUrl() != null) {
            deviceDestroyBill.setDestroyFileUrl(uploadDestroyFileVo.getDestroyFileUrl());
        }
        if (uploadDestroyFileVo.getDestroyFileName() != null) {
            deviceDestroyBill.setDestroyFileName(uploadDestroyFileVo.getDestroyFileName());
        }
        if (uploadDestroyFileVo.getUndertaker() != null) {
            deviceDestroyBill.setUndertaker(uploadDestroyFileVo.getUndertaker());
        }
        if (uploadDestroyFileVo.getLeader() != null) {
            deviceDestroyBill.setLeader(uploadDestroyFileVo.getLeader());
        }
        if (uploadDestroyFileVo.getSupervisor() != null) {
            deviceDestroyBill.setSupervisor(uploadDestroyFileVo.getSupervisor());
        }
        deviceDestroyBillService.updateEntity(deviceDestroyBill);
        //任务完结
        taskService.moveToEnd(taskService.addInvolveUser(taskBto,userUtils.getCurrentUserId()));
        //生成日志所需的文件列表
        ArrayList<FileVo> billList = ListUtil.createList(
                deviceDestroyBill.getDestroyFileName() != null && deviceDestroyBill.getDestroyFileUrl() != null ?
                        new FileVo("销毁单据", deviceDestroyBill.getDestroyFileName(), deviceDestroyBill.getDestroyFileUrl()) : null
        );
        List<Integer> list = StringSplitUtil.split(deviceDestroyBill.getDestroyDeviceDetail());
        deviceLibraryDao.upDateLeftStatusAndOwnUnitName(DeviceLifeStatus.DESTROYED.id,"",list);
        for (Integer id : list) {
            if (id > 0) {
                //生成装备日志
                deviceLogService.addLog(new DeviceLogDto(id, "上传销毁单", billList));
            }
        }
        //发送阅知信息
        List<Integer> userIds = userPublicService.findOtherUser(userUtils.getCurrentUserId());
        MessageBto messageBto = new MessageBto(taskBto.getId(),taskBto.getBusinessType(),"业务办结",userIds,1);
        messageService.add(messageBto);
        return ResponseEntity.ok("上传成功");
    }

    private void handleDeviceDetail(DeviceDestroyBill deviceDestroyBill, ArrayList<FileVo> billList, Integer flag) {
        List<Integer> list = StringSplitUtil.split(deviceDestroyBill.getDestroyDeviceDetail());
        for (Integer id : list) {
            if (id > 0) {
                if (flag == 0) {
                    //将装备装备变更为销毁
                    DeviceLibrary deviceLibraryEntity = deviceLibraryService.getOne(id);
                    deviceLibraryEntity.setLifeStatus(DeviceLifeStatus.IN_THE_DESTROYED.id);
                    deviceLibraryService.update(deviceLibraryEntity);
                }
                //生成装备日志
                deviceLogService.addLog(new DeviceLogDto(id, flag == 0 ? "装备销毁成功" : "装备销毁失败", billList));
            }
        }
        //添加出库白名单
        inputOutputDeviceService.addWhiteDevices(list,userUtils.getCurrentUnitId(),1);
    }

    private String makeDocNumber(DeviceDestroyBill deviceDestroyBill) {
        if (deviceDestroyBill.getDestroyTime() != null) {
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(deviceDestroyBill.getDestroyTime());
            int year = calendar.get(Calendar.YEAR);
            return "NO:第" + year + "XH" + deviceDestroyBill.getDocNumber() + "号";
        } else {
            return "";
        }
    }

    private TaskBto initTaskBto(DeviceDestroyBill destroyBill) {

        //生成Task
        TaskBto taskBto = new TaskBto(
                StatusEnum.WAIT_UPLOAD_BILL_FILE.id,
                "销毁业务",
                0,
                ".",
                destroyBill.getId(),
                BusinessEnum.DESTROY.id,
                userUtils.getCurrentUnitId(),
                1,
                "",
                ListUtil.createList(destroyBill.getStartUserAId(), 0));
        return taskBto;
    }

}
