package com.tykj.dev.statistical.controller;

/**
 * @author zjm
 * @version 1.0.0
 * @ClassName StatisticalController.java
 * @Description TODO
 * @createTime 2020年10月16日 14:53:00
 */

import com.tykj.dev.config.swagger.AutoDocument;
import com.tykj.dev.device.library.service.DeviceLibraryService;
import com.tykj.dev.device.library.subject.domin.DeviceLibrary;
import com.tykj.dev.device.library.subject.vo.DeviceLibrarySelectVo;
import com.tykj.dev.misc.exception.ApiException;
import com.tykj.dev.misc.utils.PageUtil;
import com.tykj.dev.misc.utils.ResultUtil;
import com.tykj.dev.statistical.cache.StatisticalCache;
import com.tykj.dev.statistical.service.StatisticalService;
import com.tykj.dev.statistical.vo.DeviceStatistics;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.groupingBy;

@RestController
@Api(tags = "统计业务模块", description = "统计相关接口")
@AutoDocument
@RequestMapping("/statistical")
public class StatisticalController {

    @Autowired
    private StatisticalService statisticalService;

    @Autowired
    private StatisticalCache statisticalCache;

    @Autowired
    private DeviceLibraryService deviceLibraryService;

//    @Autowired
//    private ConfigCache configCache;

    /**
     * 查询工作台统计
     * @return 作台统计对象集合
     */
    @ApiOperation(value = "查询工作台统计")
    @GetMapping("/workBench")
    public ResponseEntity workBench(){
        return ResponseEntity.ok(statisticalService.getWorkBenchStat());
    }

    /**
     * 查询各市的装备数量
     * @return 各市装备对象集合
     */
    @ApiOperation(value = "查询各市的装备数量")
    @GetMapping("/devNum")
    public ResponseEntity selectDevNum(){
        return ResponseEntity.ok(statisticalCache.getDevNums());
    }

    /**
     * 查询全省或各市装备生命状态数量
     * @return 全省或各市装备生命状态对象集合
     */
    @ApiOperation(value = "查询全省或各市装备生命状态数量(type:1全省，2全市)")
    @GetMapping("/StateofLife/{type}")
    public ResponseEntity selectStateofLife(@PathVariable Integer type){
        if (type==1){
            return ResponseEntity.ok(statisticalCache.getProvinceDevLifeSectors());
        }
        else if(type==2){
            return ResponseEntity.ok(statisticalCache.getCityDevLifeSectors());
        }
        else {
            throw new ApiException("type只能为1或2");
        }
    }

    /**
     * 查询当前季度各市自查情况
     * @return 各市自查状态对象集合
     */
    @ApiOperation(value = "查询当前季度各市自查情况")
    @GetMapping("/selfInspection")
    public ResponseEntity selectSelfInspection(){
        return ResponseEntity.ok(statisticalCache.getSelfInspections());
    }

    /**
     * 查询年度各季度各市自查情况
     * @return 各市自查状态对象集合
     */
    @ApiOperation(value = "查询年度各季度各市自查情况")
    @GetMapping("/selfInspection/year")
    public ResponseEntity selectYearSelfInspection(){
        return ResponseEntity.ok(statisticalCache.getYearSelfInspections());
    }

    @ApiOperation(value = "查询各市最近一次自查情况")
    @GetMapping("/selfCheckStat")
    public ResponseEntity selfCheckStat(){
        return ResponseEntity.ok(statisticalCache.getSelfCheckVos());
    }

    /**
     * 查询各市核查情况数量
     * @return 各市核查状态对象集合
     */
    @ApiOperation(value = "查询各市核查情况数量")
    @GetMapping("/confirmInspection/{type}")
    public ResponseEntity selectVerification(@PathVariable Integer type) throws ParseException{
        if (type==1){
            return ResponseEntity.ok(statisticalCache.getFirstYearVerifications());
        }
        else if(type==2){
            return ResponseEntity.ok(statisticalCache.getLastYearVerifications());
        }
        else {
            throw new ApiException("type只能为1或2");
        }
    }

    /**
     * 查询专管员概述
     * @return 各市装备生命状态对象集合
     */
    @ApiOperation(value = "查询专管员概述")
    @GetMapping("/generalSituation")
    public ResponseEntity selectGeneralSituation(){
        return ResponseEntity.ok(statisticalCache.getUserGeneralSituation());
    }

    /**
     * 查询告警情况
     * @return 全省告警统计对象集合
     */
    @ApiOperation(value = "查询告警情况")
    @GetMapping("/alarmSituation")
    public ResponseEntity selectAlarmSituation(){
        return ResponseEntity.ok(statisticalCache.getAlarmSituation());
    }

    /**
     * 查询各市告警情况
     * @return 各市告警统计对象集合
     */
    @ApiOperation(value = "查询各市告警情况")
    @GetMapping("/alarmSituation/detail/{type}")
    public ResponseEntity selectAlarmSituationDetail(@PathVariable("type") int type){
        if (type==1){
            return ResponseEntity.ok(statisticalCache.getQuarterAlarmSituations());
        }
        else if(type==2){
            return ResponseEntity.ok(statisticalCache.getYearAlarmSituations());
        }
        else {
            throw new ApiException("type只能为1或2");
        }
    }

    /**
     * 查询业务情况
     * @return 各市装备生命状态对象集合
     */
    @ApiOperation(value = "查询业务情况")
    @GetMapping("/businessSituation/{type}")
    public ResponseEntity selectBusinessSituation(@PathVariable Integer type) throws ParseException {
        return ResponseEntity.ok(statisticalService.getBusinessNum(type));
    }

    /**
     * 查询各市最近一次自查核查数量
     * @return 各市自查核查数量对象集合
     */
    @ApiOperation(value = "查询各市最近一次自查核查数量")
    @GetMapping("/checkNum")
    public ResponseEntity selectCheckNum(){
        return ResponseEntity.ok(statisticalCache.getChecks());
    }

    @ApiOperation(value = "查询所有装备统计列表", notes = "可以通过这个接口查询所有装备统计列表")
    @PostMapping("/device")
    public ResponseEntity selectAllDeviceStatList(@RequestBody DeviceLibrarySelectVo deviceLibrarySelectVo) {
        List<DeviceLibrary> deviceLibraryEntities = deviceLibraryService.getAllList(deviceLibrarySelectVo);
        deviceLibraryEntities.forEach(DeviceLibrary::setConfigName);
        Page<DeviceLibrary> deviceLibraries = PageUtil.getPerPage(deviceLibrarySelectVo.getPage(),deviceLibrarySelectVo.getSize(),deviceLibraryEntities,deviceLibrarySelectVo.getPageable());
        Map<String,Object> map = new HashMap<>();
//        Map<Integer,String> lifeStatusMap = configCache.getLifeStatusMap();
//        Set<Integer> status = deviceLibraryEntities.stream().map(DeviceLibrary::getLifeStatus).collect(Collectors.toSet());
        map.put("pages",deviceLibraries);
        map.put("models",deviceLibraryEntities.stream().map(DeviceLibrary::getModel).collect(Collectors.toSet()));
        map.put("names",deviceLibraryEntities.stream().map(DeviceLibrary::getName).collect(Collectors.toSet()));
//        map.put("ownUnits",deviceLibraryEntities.stream().map(DeviceLibrary::getOwnUnit).collect(Collectors.toSet()));
//        map.put("locationUnits",deviceLibraryEntities.stream().map(DeviceLibrary::getLocationUnit).collect(Collectors.toSet()));
//        map.put("lifeStatus",status.stream().map(integer -> new LifeStatusVo(integer,lifeStatusMap.get(integer))).collect(Collectors.toList()));
        return ResultUtil.success(map);
    }

    /**
     * 按型号，名称，类型组合
     */
    private List<DeviceStatistics> getDeviceStatistics(List<DeviceLibrary> deviceLibraries) {
        List<DeviceStatistics> deviceStatisticsList = new ArrayList<>();
        Map<String, List<DeviceLibrary>> map = deviceLibraries.stream().collect(groupingBy(deviceLibrary -> deviceLibrary.getModel()+"Ǵ"+deviceLibrary.getName()));
        for (String s:map.keySet()) {
            String[] strings = s.split("Ǵ");
            if (strings.length == 2) {
                DeviceStatistics deviceStatistics = new DeviceStatistics();
                deviceStatistics.setModel(strings[0]);
//                deviceStatistics.setName(strings[1]);
//                if ("null".equals(strings[1])){
//                    deviceStatistics.setType(null);
//                    deviceStatistics.setTypeName("-");
//                }
//                else {
//                    deviceStatistics.setType(Integer.valueOf(strings[1]));
//                    deviceStatistics.setTypeName(map.get(s).get(0).setConfigName().getTypeName());
//                }
                deviceStatistics.setName(strings[1]);
                deviceStatistics.setNum(map.get(s).size());
                deviceStatistics.setDeviceIds(map.get(s).stream().map(DeviceLibrary::getId).collect(Collectors.toList()));
                deviceStatisticsList.add(deviceStatistics);
            }
        }
        return deviceStatisticsList;
    }
}
