package com.tykj.dev.device.task.subject.domin;

import com.tykj.dev.device.task.subject.bto.TaskBto;
import com.tykj.dev.misc.base.BeanHelper;
import com.tykj.dev.misc.utils.StringSplitUtil;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.modelmapper.ModelMapper;
import org.springframework.data.annotation.CreatedBy;
import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedBy;
import org.springframework.data.annotation.LastModifiedDate;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import javax.persistence.*;
import java.util.Date;

/**
 * entity class for task
 * 任务详情
 */
@Data
@AllArgsConstructor
@NoArgsConstructor
@Entity
@ApiModel("任务详情")
@EntityListeners(AuditingEntityListener.class)
@SQLDelete(sql = "update task set delete_tag = 1 where id = ?")
@Where(clause = "delete_tag = 0")
public class Task {

    @Id
    @GeneratedValue
    @ApiModelProperty(value = "主键id")
    @Column(columnDefinition = "integer NOT NULL AUTO_INCREMENT")
    private Integer id;

    @ApiModelProperty(value = "任务状态")
    private Integer billStatus;

    @ApiModelProperty(value = "任务标题")
    private String title;

    @CreatedBy
    @ApiModelProperty(value = "创建用户id")
    private Integer createUserId;

    @CreatedDate
    @ApiModelProperty(value = "创建时间")
    private Date createTime;

    @LastModifiedBy
    @ApiModelProperty(value = "更新用户id")
    private Integer updateUserId;

    @LastModifiedDate
    @ApiModelProperty(value = "更新时间")
    private Date updateTime;

    @ApiModelProperty(value = "父任务主键id")
    private Integer parentTaskId;

    @ApiModelProperty(value = "所有节点任务id组合,例如.1.2.,意为从根节点到当前节点所经历的节点为1和2")
    private String nodeIdDetail;

    @ApiModelProperty(value = "对应账单ID")
    private Integer billId;

    @ApiModelProperty(value = "删除标记(0:未删除，1:已删除)")
    private Integer deleteTag = 0;

    @ApiModelProperty(value = "业务类型主键id")
    private Integer businessType;

    @ApiModelProperty(value = "所属单位")
    private Integer ownUnit;

    @Column(name = "top_flag_detail",columnDefinition = "TEXT")
    @ApiModelProperty(value = "用户置顶信息")
    private String topFlagDetail;

    @Column(name = "user_read_detail",columnDefinition = "TEXT")
    @ApiModelProperty(value = "用户阅读信息")
    private String userReadDetail;

    @ApiModelProperty(value = "数字类型，用于代表指针目前的下标")
    private Integer currentPoint;

    @Column(name = "custom_info",columnDefinition = "TEXT")
    @ApiModelProperty(value = "自定义信息，针对不同业务需要保存一些自定信息")
    private String customInfo;

    @Column(name = "involve_users",columnDefinition = "TEXT")
    @ApiModelProperty(value = "工作涉及人员id(x作为分隔符),例如x1x2x,意为id为1,2的用户参与了该任务")
    private String involveUsers;

    @ApiModelProperty(value = "区块链记录id")
    private String recordId;

    @ApiModelProperty(value = "备注")
    private String remark;

    public Task(String title, Integer parentTaskId, String nodeIdDetail, Integer businessType, Integer billId, Integer ownUnit) {
        this.title = title;
        this.parentTaskId = parentTaskId;
        this.nodeIdDetail = nodeIdDetail;
        this.businessType = businessType;
        this.billId = billId;
        this.ownUnit = ownUnit;
    }

    public Task(Integer status, String title, Integer parentTaskId, String nodeIdDetail, Integer businessType, Integer billId, Integer ownUnit) {
        this.billStatus = status;
        this.title = title;
        this.parentTaskId = parentTaskId;
        this.nodeIdDetail = nodeIdDetail;
        this.businessType = businessType;
        this.billId = billId;
        this.ownUnit = ownUnit;
        this.currentPoint = 0;
    }

    /**
     * do类转化为bto类
     */
    public TaskBto parse2Bto() {
        //modelMap复制
        ModelMapper mapper = BeanHelper.getUserMapper();
        TaskBto taskBto = mapper.map(this, TaskBto.class);
        //解析组合字段并添加
        taskBto.setInvolveUserIdList(StringSplitUtil.userIdSplit(this.involveUsers));
        taskBto.setTopFlagDetailList(StringSplitUtil.userIdSplit(this.topFlagDetail));
        taskBto.setUserReadDetailList(StringSplitUtil.userIdSplit(this.userReadDetail));
        return taskBto;
    }

    public Task copy(){
        //modelMap复制
        ModelMapper mapper = BeanHelper.getUserMapper();
        Task task = mapper.map(this, Task.class);
        this.createTime = new Date();
        this.updateTime = new Date();
        return task;
    }
}
