package com.tykj.dev.device.confirmcheck.entity.domain;

import com.tykj.dev.device.library.subject.domin.DeviceLibrary;
import com.tykj.dev.misc.utils.TimestampUtil;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.springframework.data.annotation.CreatedBy;
import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedBy;
import org.springframework.data.annotation.LastModifiedDate;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.persistence.*;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * entity class for device_check_detail
 * 核查详情
 */
@Data
@Entity
@EntityListeners(AuditingEntityListener.class)
@SQLDelete(sql = "update device_check_detail set delete_tag = 1 where id = ?")
@Where(clause = "delete_tag = 0")
@ApiModel("核查详情")
@Table(name = "device_check_detail", schema = "device_check_detail")
@NoArgsConstructor
public class DeviceCheckDetailEntity {

    /**
     * 主键id
     */
    @Id
    @GeneratedValue(strategy = GenerationType.IDENTITY)

    @ApiModelProperty(name = "主键id")
    private Integer id;
    /**
     * 核查账单id
     */

    @ApiModelProperty(value = "核查账单id")
    private Integer finalCheckId;
    /**
     * 检查组成员1(省A岗)id
     */

    @ApiModelProperty(value = "检查组成员1(省A岗)id")
    @Column(name = "check_userA_id")
    private Integer checkUserAId = 0;
    /**
     * 检查组成员2(省A岗)id
     */

    @ApiModelProperty(value = "检查组成员2(省A岗)id")
    @Column(name = "check_userB_id")
    private Integer checkUserBId = 0;
    /**
     * 本级经办人id(A岗)
     */

    @ApiModelProperty(value = "本级经办人id(A岗)")
    @Column(name = "userA_id")
    private Integer userAId;
    /**
     * 本级审核人id(B岗)
     */

    @ApiModelProperty(value = "本级审核人id(B岗)")
    @Column(name = "userB_id")
    private Integer userBId;
    /**
     * 核查时间
     */

    @ApiModelProperty(value = "核查时间")
    private Date checkTime;
    /**
     * 核查单位
     */

    @ApiModelProperty(value = "核查单位")
    private String checkUnit;
    /**
     * 应查数量
     */

    @ApiModelProperty(value = "应查数量")
    private Integer checkingCount;
    /**
     * 实查数量
     */

    @ApiModelProperty(value = "实查数量")
    private Integer checkedCount;
    /**
     * 核查结果
     */

    @ApiModelProperty(value = "核查结果")
    private String checkResult;
    /**
     * 核查详情(装备主键id+核对结果(0缺失1无误2新增,字符-作为状态分隔符字符,作为分隔符))
     */

    @ApiModelProperty(value = "核查详情(装备主键id+核对结果(0缺失1无误2新增3不在库9未检查,字符x作为分隔符)),例如1-2,2-2,意为主键id为1的装备缺失,为2的无误,为3的新增")
    private String checkDetail;
    /**
     * 核查状态(0:待核查,1:审核失败,2:核查完成)
     */

    @ApiModelProperty(value = "核查状态(0:待核查,1:审核失败,2:核查完成)")
    private Integer checkStatus;
    /**
     * 创建用户id
     */
    @CreatedBy

    @ApiModelProperty(value = "创建用户id")
    private Integer createUserId;
    /**
     * 创建时间
     */
    @CreatedDate

    @ApiModelProperty(value = "创建时间")
    private Date createTime;
    /**
     * 更新用户id
     */
    @LastModifiedBy

    @ApiModelProperty(value = "更新用户id")
    private Integer updateUserId;
    /**
     * 更新时间
     */
    @LastModifiedDate

    @ApiModelProperty(value = "更新时间")
    private Date updateTime;
    /**
     * 删除标记(0:未删除，1:已删除)
     */

    @ApiModelProperty(value = "删除标记(0:未删除，1:已删除)")
    private Integer deleteTag = 0;
    /**
     * 预留字段1
     */

    @ApiModelProperty(value = "备注")
    private String remark;
    /**
     * 预留字段2
     */

    @ApiModelProperty(value = "预留字段2")
    private String var2;
    @ApiModelProperty(value = "核查标题")
    @Transient
    private String title;
    @ApiModelProperty(value = "创建人")
    @Transient
    private String createUser;
    @ApiModelProperty(value = "核查人")
    @Transient
    private String checkUser;
    @ApiModelProperty(value = "审核人")
    @Transient
    private String confirmUser;

    /**
     * 用于生成市级的核查详情账单的构造函数
     */
    public DeviceCheckDetailEntity(Integer finalCheckId, Date checkTime, String checkUnit, Integer checkingCount, Integer checkedCount, String checkResult, String checkDetail, Integer checkStatus) {
        this.finalCheckId = finalCheckId;
        this.checkTime = checkTime;
        this.checkUnit = checkUnit;
        this.checkingCount = checkingCount;
        this.checkedCount = checkedCount;
        this.checkResult = checkResult;
        this.checkDetail = checkDetail;
        this.checkStatus = checkStatus;
    }

    /**
     * 用于生成县级的核查详情账单的构造函数
     */
    public DeviceCheckDetailEntity(Integer finalCheckId,
                                   Integer checkUserAId,
                                   Integer checkUserBId,
                                   Integer userAId,
                                   Integer userBId,
                                   Date checkTime,
                                   String checkUnit,
                                   Integer checkingCount,
                                   Integer checkedCount,
                                   String checkResult,
                                   String checkDetail,
                                   Integer checkStatus) {
        this.finalCheckId = finalCheckId;
        this.checkUserAId = checkUserAId;
        this.checkUserBId = checkUserBId;
        this.userAId = userAId;
        this.userBId = userBId;
        this.checkTime = checkTime;
        this.checkUnit = checkUnit;
        this.checkingCount = checkingCount;
        this.checkedCount = checkedCount;
        this.checkResult = checkResult;
        this.checkDetail = checkDetail;
        this.checkStatus = checkStatus;
    }

    /**
     * @param checkUnit   要核查的单位
     * @param goodDevices 所属与所在均在本单位的装备集合
     * @param badDevices  所属在，但所在不在本单位的装备集合
     * @return 初始化的账单(还没有被人修改过的)
     */
    public static DeviceCheckDetailEntity Empty(String checkUnit, List<DeviceLibrary> goodDevices, List<DeviceLibrary> badDevices) {
        //构造checkDetail 分当前在库与不在库的 赋予不同状态
        String goodCheckDetail = goodDevices.stream()
                .map(DeviceLibrary::getId)
                .map(Objects::toString)
                .collect(Collectors.joining("-2,"));

        String badCheckDetail = badDevices.stream()
                .map(DeviceLibrary::getId)
                .map(Objects::toString)
                .collect(Collectors.joining("-3,"));

        //如果不在库的不为空，则拼接，否则没必要
        String checkDetail = StringUtils.isEmpty(badCheckDetail) ? goodCheckDetail : goodCheckDetail + "," + badCheckDetail;

        return new DeviceCheckDetailEntity(
                0,
                0,
                0,
                0,
                0,
                TimestampUtil.getNowDate(),
                checkUnit,
                goodDevices.size(),
                0,
                "",
                checkDetail,
                0

        );
    }

    public static DeviceCheckDetailEntity EmptyWithChecker(
            Integer checkUserA,
            Integer checkUserB,
            Integer userAId,
            Integer userBId,
            String checkUnit,
            List<DeviceLibrary> goodDevices,
            List<DeviceLibrary> badDevices) {
        //构造checkDetail 分当前在库与不在库的 赋予不同状态
        String goodCheckDetail = "";
        if (!CollectionUtils.isEmpty(goodDevices)) {
            goodCheckDetail = goodDevices.stream()
                    .map(device -> device.getId() + "-9")
                    .collect(Collectors.joining(","));
        }

        String badCheckDetail = "";
        if (!CollectionUtils.isEmpty(badDevices)) {
            badCheckDetail = badDevices.stream()
                    .map(device -> device.getId() + "-3")
                    .collect(Collectors.joining(","));
        }


        //如果不在库的不为空，则拼接，否则没必要
        String checkDetail = StringUtils.isEmpty(badCheckDetail) ? goodCheckDetail : goodCheckDetail + "," + badCheckDetail;

        return new DeviceCheckDetailEntity(
                0,
                checkUserA,
                checkUserB,
                userAId,
                userBId,
                TimestampUtil.getNowDate(),
                checkUnit,
                goodDevices.size(),
                0,
                "",
                checkDetail,
                0

        );
    }
}
