package entity.domain;

import com.tykj.dev.device.user.subject.entity.User;
import entity.enums.DestroyStatus;
import entity.vo.DeviceDestroyFormVo;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.springframework.data.annotation.CreatedBy;
import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedBy;
import org.springframework.data.annotation.LastModifiedDate;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import javax.persistence.*;
import java.util.Date;
import java.util.List;

/**
 * entity class for device_destroy_bill
 * 装备销毁账单
 * @author huangxiahao
 */
@Data
@Entity
@EntityListeners(AuditingEntityListener.class)
@SQLDelete(sql = "update device_destroy_bill set delete_tag = 1 where id = ?")
@Where(clause = "delete_tag = 0")
@ApiModel("装备销毁账单")
public class DeviceDestroyBill {

	
	/**
	* 主键id
	*/
	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	@ApiModelProperty(name = "主键id")
	private Integer id ;	
	/**
	* 发起销毁业务经办人id(A岗)
	*/
	@ApiModelProperty(value = "发起销毁业务经办人id(A岗)")
	private Integer startUserAId ;	
	/**
	* 发起销毁业务审核人id(B岗)
	*/
	@ApiModelProperty(value = "发起销毁业务审核人id(B岗)")
	private Integer startUserBId ;	
	/**
	* 销毁状态(0:销毁装备出库待审核,1:销毁装备出库审核失败,2:已销毁)
	*/
	@ApiModelProperty(value = "销毁状态(0:销毁装备出库待审核,1:销毁装备出库审核失败,2:已销毁)")
	private Integer destroyStatus ;
	/**
	* 出库附件文件名
	*/
	@ApiModelProperty(value = "出库附件文件名")
	private String fileName ;	
	/**
	* 出库附件文件地址URL
	*/
	@ApiModelProperty(value = "出库附件文件地址URL")
	private String fileUrl ;	
	/**
	* 装备销毁出库详情(装备主键id+核对结果(0缺失1无误2新增,字符x作为分隔符)),例如10x21x32,意为主键id为1的装备缺失,为2的无误,为3的新增
	*/
	@ApiModelProperty(value = "装备销毁出库详情(装备主键id+核对结果(0缺失1无误2新增,字符x作为分隔符)),例如10x21x32,意为主键id为1的装备缺失,为2的无误,为3的新增")
	private String destroyDeviceDetail ;
	/**
	* 创建用户id
	*/
	@ApiModelProperty(value = "创建用户id")
	@CreatedBy
	private Integer createUserId ;	
	/**
	* 创建时间
	*/
	@ApiModelProperty(value = "创建时间")
	@CreatedDate
	private Date createTime ;
	/**
	* 更新用户id
	*/
	@ApiModelProperty(value = "更新用户id")
	@LastModifiedBy
	private Integer updateUserId ;
	/**
	* 更新时间
	*/
	@ApiModelProperty(value = "更新时间")
	@LastModifiedDate
	private Date updateTime ;
	/**
	* 删除标记(0:未删除，1:已删除)
	*/
	@ApiModelProperty(value = "删除标记(0:未删除，1:已删除)")
	private Integer deleteTag  = 0 ;
	/**
	* 文号
	*/
	@ApiModelProperty(value = "文号")
	private Integer docNumber ;
	/**
	* 监销人
	*/
	@ApiModelProperty(value = "监销人")
	private String supervisor ;
	/**
	* 主管领导
	*/
	@ApiModelProperty(value = "主管领导")
	private String leader ;
	/**
	* 承办人
	*/
	@ApiModelProperty(value = "承办人")
	private String undertaker ;
	/**
	* 销毁日期
	*/
	@ApiModelProperty(value = "销毁日期")
	private Date destroyTime ;

	@ApiModelProperty(value = "经办人")
	@Transient
	private String userA ;

	@ApiModelProperty(value = "型号")
	@Transient
	private String model ;

	@ApiModelProperty(value = "名称")
	@Transient
	private String name ;

	public DeviceDestroyBill(Integer startUserAId, Integer startUserBId, Integer destroyStatus, String fileName, String fileUrl, String supervisor, String leader, String undertaker, String userA, String destroyDeviceDetail) {
		this.startUserAId = startUserAId;
		this.startUserBId = startUserBId;
		this.destroyStatus = destroyStatus;
		this.fileName = fileName;
		this.fileUrl = fileUrl;
		this.supervisor = supervisor;
		this.leader = leader;
		this.undertaker = undertaker;
		this.userA = userA;
	}

	public static DeviceDestroyBill formVoToBill(DeviceDestroyFormVo formVo, User user){
		return new DeviceDestroyBill(
				user.getUserId(),
				formVo.getConfirmUserId(),
				DestroyStatus.NEED_CONFIRM.getStatus(),
				formVo.getFileName(),
				formVo.getFileUrl(),
				formVo.getSupervisor(),
				formVo.getLeader(),
				formVo.getUndertaker(),
				user.getName(),
				deviceDetailListToString(formVo.getDevices())
		);
	}

	/**
	 * 生成装备销毁出库详情
	 *
	 * @return 装备出库详情
	 */
	public static String deviceDetailListToString(List<Integer> deviceIds) {
		StringBuffer deviceDetail = new StringBuffer();
		deviceDetail.append("x");
		deviceIds.forEach(device -> {
			//拼接对应的字符串  例如: 21x
			deviceDetail.append(device.toString());
			deviceDetail.append(1);
			deviceDetail.append("x");
		});
		return deviceDetail.toString();
	}
}
