package com.zjty.fp.acq.pssp.subject.service.impl;

import com.google.common.collect.Lists;
import com.zjty.fp.acq.misc.entity.ImportAds;
import com.zjty.fp.acq.misc.utils.DateTimeUtil;
import com.zjty.fp.acq.misc.utils.FileCreator;
import com.zjty.fp.acq.misc.utils.JacksonUtil;
import com.zjty.fp.acq.pssp.subject.entity.TimeTup;
import com.zjty.fp.acq.pssp.subject.entity.location.Alert;
import com.zjty.fp.acq.pssp.subject.entity.remote.RemoteAlert;
import com.zjty.fp.acq.pssp.subject.repository.remote.RemoteAlertRepository;
import com.zjty.fp.acq.pssp.subject.service.AlertService;
import com.zjty.fp.acq.pssp.task.CollectDataTask;
import com.zjty.fp.acq.pssp.utils.DicMapUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.UrlResource;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.transaction.Transactional;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.time.LocalDate;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.zjty.fp.acq.misc.entity.PsspCount.COUNT_ADDRESS_ALERT;
import static com.zjty.fp.acq.pssp.task.CollectDataTask.dynamicNow;
import static java.util.stream.Collectors.toList;

/**
 * 告警服务实现类
 *
 * @author : Matrix [xhyrzldf@foxmail.com]
 * date  : 18-8-21
 */
@Slf4j
@Service
public class AlertServiceImpl implements AlertService {

    private static final String alertName = "alert";
    private static String rootURL = "21.28.120.10";
    @Autowired
    FileCreator fileCreator;
    @Autowired
    private RemoteAlertRepository remoteAlertRepo;
    private long supposeId = 0;

    /**
     * 抓取所有源数据然后保存入库,历史任务进行的时候，更新任务暂停
     * 注意 : 这是一个相当耗时的操作
     * 历史数据也不用写入文件
     */
    @Override
    public void fetchAllData(TimeTup timeTup) {


        log.info("[pssp] [历史任务] 正在进行历史任务采集任务... 要采集的月份时间为{}", timeTup.getStartTime());

        //该任务是一个异步任务
        CompletableFuture.runAsync(() -> {


            int handledDataCount = 0;
            try {
                //先将其他数据同步计划任务关闭
                CollectDataTask.trigger = false;
                log.info("[pssp] [历史任务] 采集 {} 时间的历史数据 ", timeTup.getLocalStartTime().toString());
                //设置远程表的动态月份

                CollectDataTask.setDynamicTableMonth(timeTup.getLocalStartTime().toLocalDate());

                log.info("[pssp] 准备抓取源报警历史数据");
                //历史数据这里的操作分块操作,每10W条历史数据进行一次分块
                long dataCount = remoteAlertRepo.findDataFromTime(timeTup.getStartTime(), timeTup.getEndTime()).size();
                //如果数据量大于10w 分多次操作
                int size = 50000;
                log.info("[pssp] 报警数据历史同步任务数据量较大,数据量为 {},进行分块处理", dataCount);
                long jobCount = (dataCount - 1) / size + 1;
                for (int page = 0; page < jobCount; page++) {
                    //findAll(new PageRequest(page, size)).getContent();
                    // 采集原始数据
                    List<RemoteAlert> content = remoteAlertRepo.findDataFromTime(timeTup.getStartTime(), timeTup.getEndTime());
                    List<Alert> partAlert = content
                            .parallelStream()
                            .map(RemoteAlert::toDo)
                            .collect(toList());
                    log.info("[pssp] 第{}/{}部分的历史数据[采集]完成,该部分处理了{}条数据", page + 1, jobCount, partAlert.size());

                    // 下载
                    for (Alert alert : partAlert) {
                        if (!StringUtils.isEmpty(alert.getWebSnapshot())) {
                            //下载web快照,解析url
                            downloadWeb(alert);
                        }

                        if (!StringUtils.isEmpty(alert.getDocSnapshot())) {
                            //下载doc快照,解析url
                            downloadDoc(alert);
                        }
                    }

                    handledDataCount += partAlert.size();
                }
                log.info("[pssp] {} 月源报警历史附件抓取任务完成,size = {}", timeTup.getLocalStartTime(), handledDataCount);
            } catch (Exception e) {
                log.error("[pssp] [历史任务] 采集发生异常,目前采集月份为 {},采集数量为 {}, 异常栈 : {}", timeTup.getLocalStartTime(), handledDataCount, e);
            } finally {
                //历史任务完成之后重新设置回当前月份
                log.info("[pssp] [历史任务] {} 月采集任务结束,将月份重新设置回{}时间", timeTup.getLocalStartTime(), LocalDate.now());
                CollectDataTask.setDynamicTableMonth(LocalDate.now());
                //4.更新 mcMap(如果mcMap中存在当前月的key-value值) 2021-4
                Map<String, Long> mcMap = DicMapUtil.readMonthFile();
                String currentKey = LocalDate.now().getYear() + "-" + LocalDate.now().getMonthValue();
                if (mcMap.containsKey(currentKey)) {
                    //更新count值
                    long cmc = remoteAlertRepo.count();
                    mcMap.put(currentKey, cmc);
                    DicMapUtil.updateMonthFile(mcMap);
                    log.info("[pssp] [alert] mcMap中存在当前月: {} 的记录，更新count值 = {}", currentKey, cmc);
                }
                //全部完成后再开启计划任务
                CollectDataTask.trigger = true;
            }
        });

    }


    /**
     * 获取所有更新的数据,步骤如下
     * <li>1.获取当前月份本地最新的一条报警数据id</li>
     * <li>2.向源数据库抓取在步骤1id之后的数据</li>
     * <li>3.数据入库</li>
     * <li>4.更新缓存相关数据</>
     */
    @Override
    public void fetchUpdatedData() {

        //从拦截器中获得当前应当处理的月份时间,并进行查询获得当前月份本地数据库报警数据的最新id
        //从dynamicMonth中获取当前应处理的时间，读取对应文件中的最新id值
        String key = dynamicNow.getYear() + "-" + dynamicNow.getMonthValue();
        Map<String, Integer> monthMap = DicMapUtil.readDictFile(alertName);
        Integer lastedDataId = monthMap.getOrDefault(key, 0);

        log.info("[pssp] [alert] 执行更新数据任务,本地数据库 {} 月最新数据的id为 {} ,向源数据库采集在这id之后的数据....", key, lastedDataId);
        log.debug("[pssp] [id_check],supposeId = {},actuallyId = {}", supposeId, lastedDataId);

        List<Alert> updatedData = remoteAlertRepo.findDataFromId(Long.valueOf(lastedDataId)).stream()
                .map(RemoteAlert::toDo)
                .collect(toList());

        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] [alert] 本次更新采集没有要更新的数据");
        } else {
            log.info("[pssp] [alert] 采集完成,本次采集了 {} 条数据,数据写入本机数据库与Es数据库", updatedData.size());

            // 每1000条分成一个任务来处理 size / 1000 = 任务总数
            supposeId = updatedData.stream()
                    .map(Alert::getId)
                    .max(Comparator.naturalOrder())
                    .orElse(0L);

            int jobCount = updatedData.size() / 1000;

            int i = 1;

            // 分批处理 数据
            for (List<Alert> alerts : Lists.partition(updatedData, 1000)) {
                log.info("[pssp]分批处理数据中... 当前批次数据量{}, 进度{}/{}", alerts.size(), i++, jobCount + 1);
                handleData(key, monthMap, alerts);
            }

            //4.更新 mcMap(如果mcMap中存在当前月的key-value值) 2021-4
            Map<String, Long> mcMap = DicMapUtil.readMonthFile();
            String currentKey = LocalDate.now().getYear() + "-" + LocalDate.now().getMonthValue();
            if (mcMap.containsKey(currentKey)) {
                //更新count值
                long cmc = remoteAlertRepo.count();
                mcMap.put(currentKey, cmc);
                DicMapUtil.updateMonthFile(mcMap);
                log.info("[pssp] [alert] mcMap中存在当前月: {} 的记录，更新count值 = {}", currentKey, cmc);
            }

        }

    }

    private void handleData(String key, Map<String, Integer> monthMap, List<Alert> updatedData) {
        //1.更新写入本地文件
        String webJson = JacksonUtil.toJSon(updatedData).replace("\n", "");
        fileCreator.createFileAndZip(COUNT_ADDRESS_ALERT, "pssp", alertName, webJson);
        //2.记住最大ID值
        long maxId = updatedData.stream().mapToLong(Alert::getId).max().orElse(0L);
        log.info("[pssp] [alert] 更新后的最大报警数据id为:{},记录到文件中", maxId);
        monthMap.put(key, (int) maxId);
        DicMapUtil.createDictFile(alertName, monthMap);
        log.info("[pssp] [alert] 报警数据更新完成");
        //3.下载文件后保存到指定目录
        //解析要下载的URL
        for (Alert alert : updatedData) {
            if (!StringUtils.isEmpty(alert.getWebSnapshot())) {
                //下载web快照,解析url
                downloadWeb(alert);
            }

            if (!StringUtils.isEmpty(alert.getDocSnapshot())) {
                //下载doc快照,解析url
                downloadDoc(alert);
            }
        }
    }

    private boolean downloadWeb(Alert alert) {
        //1. getURL
        String webUrl = "http://" + rootURL + "/snapshot" + alert.getWebSnapshot();
        log.info("[pssp] [alert] 准备下载网页快照文件到本地，id = {} , time = {} , 快照地址 : {}", alert.getId(), alert.getTmFetch(), webUrl);
        //2. download
        try {
            UrlResource resource = new UrlResource(webUrl);
            //下载完成后文件写入到 rootPath/yyyyMMdd/web/源数据id+html后缀
            //拼接出文件路径
            String dateString = DateTimeUtil.formatDateTimetoString(alert.getTmFetch(), DateTimeUtil.FMT_yyyyMMdd_8);
            String filePath = ImportAds.IMPORT_URL + dateString + "/web/";
            String fileName = alert.getId() + ".html";

            boolean make = createFilePath(new File(filePath));
            if (make) {
                try {
                    // 写入文件
                    writeToFile(filePath + fileName, resource.getInputStream());
                    log.info("[pssp] [alert] 转储快照文件成功");
                } catch (IOException e) {
                    log.info("[pssp] [alert] 生成文件时出现异常：" + e);
                }
            }

            return true;
        } catch (IOException e) {
            log.info("[pssp] [alert] 下载转储快照文件发生错误 : {}", e.toString());
            return false;
        }
    }

    private boolean downloadDoc(Alert alert) {
        //1. getURL
        String docSnapshot1;
        String docSnapshot2;
        String regexp = "/(.*).html";
        Pattern p = Pattern.compile(regexp);
        Matcher matcher = p.matcher(alert.getDocSnapshot());
        matcher.find();
        String[] snapshots = matcher.group(1).split("/");
        int indexMax = snapshots.length - 1;
        docSnapshot1 = snapshots[0];
        docSnapshot2 = snapshots[indexMax];

        String[] urlS = alert.getDocUrl().split("\\.");
        String suffix = urlS[urlS.length - 1];
        //1.拼接 - > url
        String docUrl = "http://" + rootURL + "/view/web_" + alert.getIdWeb() + "/" + docSnapshot1 + "/" + docSnapshot2 + "." + suffix;
        log.info("[pssp] [alert] 准备下载附件文件到本地，id = {} , time = {} 附件地址 : {}", alert.getId(), alert.getTmFetch(), docUrl);
        //2. download
        try {
            UrlResource resource = new UrlResource(docUrl);
            //下载完成后文件写入到 rootPath/yyyyMMdd/web/源数据id+html后缀
            //拼接出文件路径
            String dateString = DateTimeUtil.formatDateTimetoString(alert.getTmFetch(), DateTimeUtil.FMT_yyyyMMdd_8);
            String filePath = ImportAds.IMPORT_URL + dateString + "/doc/";
            String fileName = alert.getId() + "." + suffix;

            boolean make = createFilePath(new File(filePath));
            if (make) {
                try {
                    // 写入文件
                    writeToFile(filePath + fileName, resource.getInputStream());
                    log.info("[pssp] [alert] 转储附件文件成功");
                } catch (IOException e) {
                    log.info("[pssp] [alert] 生成文件时出现异常：" + e);
                }
            }

            return true;
        } catch (IOException e) {
            log.info("[pssp] [alert] 下载转储附件发生错误 : {}", e.toString());
            return false;
        }
    }

    /**
     * 生成指定的文件夹路径
     * 若文件夹不存在则创建
     */
    private boolean createFilePath(File file) {
        return file.exists() || file.mkdirs();
    }

    private void writeToFile(String filePath, InputStream stream) throws IOException {
        FileChannel outChannel = new FileOutputStream(filePath).getChannel();
        ReadableByteChannel inChannel = Channels.newChannel(stream);
        ByteBuffer buffer = ByteBuffer.allocate(1024);

        while (true) {
            if (inChannel.read(buffer) == -1) {
                break;
            }

            buffer.flip();
            outChannel.write(buffer);
            buffer.clear();
        }

        inChannel.close();
        outChannel.close();
    }


}
