package com.zjty.fp.acq.pssp.subject.service.impl;

import com.zjty.fp.acq.misc.utils.FileCreator;
import com.zjty.fp.acq.misc.utils.JacksonUtil;
import com.zjty.fp.acq.pssp.subject.entity.TimeTup;
import com.zjty.fp.acq.pssp.subject.entity.location.Website;
import com.zjty.fp.acq.pssp.subject.entity.remote.RemoteWebsite;
import com.zjty.fp.acq.pssp.subject.repository.remote.RemoteWebsiteRepository;
import com.zjty.fp.acq.pssp.subject.service.WebsiteService;
import com.zjty.fp.acq.pssp.utils.DicMapUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import static com.zjty.fp.acq.misc.entity.PsspCount.COUNT_ADDRESS_WEBSITE;
import static java.util.stream.Collectors.toList;

/**
 * fusion-platform.
 *
 * @author : Matrix [xhyrzldf@gmail.com]
 * 18-9-6 .
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class WebsiteServiceImpl implements WebsiteService {

    private static final String websiteName = "website";
    @Autowired
    FileCreator fileCreator;
    @Autowired
    private RemoteWebsiteRepository remoteWebsiteRepository;

    @Override
    public void fetchAllData(TimeTup timeTup) {
        log.info("[pssp]准备抓取全部源网站数据");
        List<Website> websiteList = remoteWebsiteRepository.findAll().stream()
                .map(RemoteWebsite::toDo)
                .collect(Collectors.toList());
        log.info("[pssp]源网站数据抓取完成,size = {},准备写入融合平台数据库", websiteList.size());
        // 异步写入数据库
        CompletableFuture.runAsync(() -> {
            log.info("[pssp]正在异步写入网站数据到本地文件");
            String webJson = JacksonUtil.toJSon(websiteList).replace("\n", "");
            log.info("[pssp]准备写入文件，要写入的文件字符长度为{}", webJson.length());
            fileCreator.createFileAndZip(COUNT_ADDRESS_WEBSITE, "pssp", "website", webJson);
            log.info("[pssp]异步网站数据写入完成");
        });
    }

    @Override
    public void fetchUpdatedData() {
        log.info("[pssp] [website] 正在执行网站数据的增量更新");
        //读取文件中的最新id , 获得当前月的时间
        Integer lastedId = DicMapUtil.readDictFile(websiteName).getOrDefault("web", 0);
        if (lastedId == 0) {
            log.info("[pssp] [website] 当前文件中没有任何网站,请先执行网站历史采集任务");
            return;
        }

        Long lastedDataId = Long.valueOf(lastedId);
        log.info("[pssp] [website] 当前本机网站表数据的最新id为 {} ,向源数据库采集在这id之后的数据....", lastedDataId);
        List<Website> updatedData = remoteWebsiteRepository.findDataFromId(lastedDataId).stream()
                .map(RemoteWebsite::toDo)
                .collect(toList());

        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] [website] 本次网站更新采集没有要更新的数据");
        } else {
            log.info("[pssp] [website] 网站数据采集完成,本次采集了 {}条数据,数据写入本机数据库", updatedData.size());
            // 异步写入数据库
            CompletableFuture.runAsync(() -> {
                log.info("[pssp] [website] 正在异步写入网站更新数据到本地文件");
                String webJson = JacksonUtil.toJSon(updatedData).replace("\n", "");
                fileCreator.createFileAndZip(COUNT_ADDRESS_WEBSITE, "pssp", "website", webJson);
                //写完要把最新的id给记录住
                long maxId = updatedData.stream().mapToLong(Website::getId).max().orElse(0L);
                log.info("[pssp] [website] 更新后的最大网站数据id为:{},记录到文件中", maxId);
                Map<String, Integer> map = new HashMap<>();
                map.put("web", (int) maxId);
                DicMapUtil.createDictFile(websiteName, map);
                log.info("[pssp] [website]异步网站更新数据写入完成");
            });
        }

    }


}
