package com.zjty.fp.acq.pssp.subject.service.impl;

import com.zjty.fp.acq.pssp.base.constant.DictConst;
import com.zjty.fp.acq.pssp.subject.entity.location.Website;
import com.zjty.fp.acq.pssp.subject.entity.remote.RemoteWebsite;
import com.zjty.fp.acq.pssp.subject.repository.location.WebsiteRepository;
import com.zjty.fp.acq.pssp.subject.repository.remote.RemoteWebsiteRepository;
import com.zjty.fp.acq.pssp.subject.service.WebsiteService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

/**
 * fusion-platform.
 *
 * @author : Matrix [xhyrzldf@gmail.com]
 * 18-9-6 .
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class WebsiteServiceImpl implements WebsiteService {

    @Autowired
    private WebsiteRepository websiteRepository;

    @Autowired
    private RemoteWebsiteRepository remoteWebsiteRepository;

    @Override
    public void fetchAllData() {
        log.info("[pssp]准备抓取全部源网站数据");
        List<Website> websiteList = remoteWebsiteRepository.findAll().stream()
                .map(RemoteWebsite::toDo)
                .collect(Collectors.toList());
        log.info("[pssp]源网站数据抓取完成,size = {},准备写入融合平台数据库", websiteList.size());
        // 异步写入数据库
        CompletableFuture.runAsync(() -> {
            log.info("[pssp]正在异步写入网站数据到本地数据库");
            websiteRepository.save(websiteList);
            log.info("[pssp]异步网站数据写入完成");

            //更新字典map
            log.info("[pssp] [定时任务] 开始更新字典Map任务,更新前的字典Map大小为,websiteMap : {}", DictConst.WEBSITE_MAP.size());
            DictConst.WEBSITE_MAP = findDictMap();
            log.info("[pssp] [定时任务] 完成更新字典Map任务,更新后的字典Map大小为,websiteMap : {}", DictConst.WEBSITE_MAP.size());
        });
    }

    @Override
    public void fetchUpdatedData() {
        log.info("[pssp]正在执行网站数据的增量更新");
        Website lastedWebsite = websiteRepository.findFirstByOrderByIdDesc();
        if (lastedWebsite == null) {
            log.info("[pssp] 当前DB中没有任何网站,请先执行网站历史采集任务");
            return;
        }

        Long lastedDataId = lastedWebsite.getId();
        log.info("[pssp]当前本机网站表数据的最新id为 {} ,向源数据库采集在这id之后的数据....", lastedDataId);
        List<Website> updatedData = remoteWebsiteRepository.findDataFromId(lastedDataId).stream()
                .map(RemoteWebsite::toDo)
                .collect(toList());

        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] 本次网站更新采集没有要更新的数据");
        } else {
            log.info("[pssp] 网站数据采集完成,本次采集了 {}条数据,数据写入本机数据库", updatedData.size());
            // 异步写入数据库
            CompletableFuture.runAsync(() -> {
                log.info("[pssp]正在异步写入网站更新数据到本地数据库");
                websiteRepository.save(updatedData);
                log.info("[pssp]异步网站更新数据写入完成");

                //更新字典map
                log.info("[pssp] [定时任务] 开始更新字典Map任务,更新前的字典Map大小为,websiteMap : {}", DictConst.WEBSITE_MAP.size());
                DictConst.WEBSITE_MAP = findDictMap();
                log.info("[pssp] [定时任务] 完成更新字典Map任务,更新后的字典Map大小为,websiteMap : {}", DictConst.WEBSITE_MAP.size());
            });
        }

    }

    @Override
    public List<Website> findAllData() {
        return websiteRepository.findAll();
    }


    @Override
    public long getCount() {
        return websiteRepository.count();
    }

    /**
     * 网站数据以字典的形式返回
     *
     * @return key : id ,value : {@link Website}
     */
    @Override
    public Map<Long, Website> findDictMap() {
        return websiteRepository.findAll()
                .stream()
                .collect(Collectors
                        .toMap(Website::getId, Function.identity()));
    }
}
