package com.zjty.fp.acq.pssp.subject.repository.location;

import com.zjty.fp.acq.pssp.subject.entity.location.Alert;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Query;
import org.springframework.stereotype.Repository;

import java.util.Date;
import java.util.List;

/**
 * @author matrix
 */
@Repository
public interface AlertRepository extends JpaRepository<Alert, Long>, JpaSpecificationExecutor<Alert> {

    /**
     * 查询最新时间的一条对象
     *
     * @return 最新时间的一条报警
     */
    Alert findFirstByOrderByTmFetchDesc();

    /**
     * 查询指定月份最新的一条报警数据,如果时间相同,取最大id
     *
     * @param specifiedMonth 例如201802,201911格式为`%Y%m`
     * @return 指定月份最新的一条报警数据
     */
    @SuppressWarnings("SqlResolve")
    @Query(value = "select * from pssp_alert p where date_format(p.tm_fetch,'%Y%m')= ?1 order by p.tm_fetch desc,p.id desc limit 1", nativeQuery = true)
    Alert findSpecifiedMonthLastedData(String specifiedMonth);

    /**
     * 查询最老时间的一条对象
     *
     * @return 最远时间距离的一条报警
     */
    Alert findFirstByOrderByTmFetch();

    /**
     * 查询指定id之后的数据
     * @param id
     * @return
     */
    Page<Alert> findByPrimaryIdAfter(Long id,Pageable pageable);

    /**
     * 查询指定时间的告警数据
     *
     * @param tmFetch 要查询的时间
     * @return 报警数据集合
     */
    List<Alert> findByTmFetch(Date tmFetch);

    /**
     * 查询今日的报警信息
     *
     * @return 今日报警数据集合
     */
    @Query(value = "SELECT * FROM pssp_alert p WHERE TO_DAYS(p.tm_fetch) = TO_DAYS(NOW())", nativeQuery = true)
    List<Alert> findTodayData();

    /**
     * 查询指定时间段内的分页告警数据
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param pageable  分页参数
     * @return 报警集合
     */
    @Query("select o from Alert o where o.tmFetch between ?1 and ?2")
    Page<Alert> findBetweenTime(Date startTime, Date endTime, Pageable pageable);

    /**
     * 查询指定时间段内的告警数据
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return 报警集合
     */
    @Query("select o from Alert o where o.tmFetch between ?1 and ?2")
    List<Alert> findBetweenTime(Date startTime, Date endTime);

    /**
     * 查询指定时间后的告警数据
     *
     * @param time 指定的时间
     * @return 报警集合
     */
    @Query("select o from Alert o where o.tmFetch >= ?1")
    List<Alert> findAfterTime(Date time);

}
