package com.zjty.fp.acq.pssp.subject.service.impl;

import com.zjty.fp.acq.pssp.subject.entity.location.Alert;
import com.zjty.fp.acq.pssp.subject.entity.remote.RemoteAlert;
import com.zjty.fp.acq.pssp.subject.repository.location.AlertRepository;
import com.zjty.fp.acq.pssp.subject.repository.remote.RemoteAlertRepository;
import com.zjty.fp.acq.pssp.subject.service.AlertService;
import com.zjty.fp.acq.pssp.task.CollectDataTask;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.security.InvalidParameterException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static com.zjty.fp.acq.misc.utils.DateTimeUtil.LocalToDate;
import static com.zjty.fp.acq.misc.utils.DateTimeUtil.parseToDate;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

/**
 * 告警服务实现类
 *
 * @author : Matrix [xhyrzldf@foxmail.com]
 * date  : 18-8-21
 */
@Slf4j
@Service
@Transactional(rollbackOn = Exception.class)
public class AlertServiceImpl implements AlertService {

    @Autowired
    private AlertRepository alertRepo;

    @Autowired
    private RemoteAlertRepository remoteAlertRepo;

    private long supposeId = 0;

    /**
     * 抓取所有源数据然后保存入库,历史任务进行的时候，更新任务暂停
     * 注意 : 这是一个相当耗时的操作
     */
    @Override
    public void fetchAllData() {
        //遍历一个时间集合,用来抓取历史数据,获取当前月,添加今年一月至当前月的数据
        List<LocalDate> monthTables = new ArrayList<>();

        //WZ项目 添加2017年12月 201801 202007 三个月的数据
        monthTables.add(LocalDate.of(2017, 12, 1));
        monthTables.add(LocalDate.of(2018, 1, 1));
//        monthTables.add(LocalDate.of(2020, 7, 1));
//        int nowYear = LocalDate.now().getYear();
//        int nowMonth = LocalDate.now().getMonthValue();
//        log.info("[pssp] [历史任务] 确认历史任务的采集月份范围");
//        for (int i = 1; i < nowMonth; i++) {
//            //check month
//            try {
//                remoteAlertRepo.count();
//                monthTables.add(LocalDate.of(nowYear, i, 1));
//            } catch (Exception e) {
//                log.warn("[pssp] [历史任务] 检测源目标数据库没有 {}-{}月表数据", nowYear, i);
//            }
//        }

        log.info("[pssp] [历史任务] 正在进行历史任务采集任务... 要采集的月份时间为{}", monthTables
                .stream()
                .map(LocalDate::toString)
                .collect(joining(",")));


        //该任务是一个异步任务
        CompletableFuture.runAsync(() -> {

            for (LocalDate date : monthTables) {

                int handledDataCount = 0;
                try {
                    //先将其他数据同步计划任务关闭
                    CollectDataTask.trigger = false;
                    log.info("[pssp] [历史任务] 采集 {} 时间的历史数据 ", date.toString());
                    //设置远程表的动态月份
                    CollectDataTask.setDynamicTableMonth(date);

                    log.info("[pssp] 准备抓取源报警历史数据");
                    //历史数据这里的操作分块操作,每10W条历史数据进行一次分块
                    long dataCount = remoteAlertRepo.count();
                    //如果数据量大于10w 分多次操作
                    int size = 50000;
                    log.info("[pssp] 报警数据历史同步任务数据量较大,数据量为 {},进行分块处理", dataCount);
                    long jobCount = (dataCount - 1) / size + 1;
                    for (int page = 0; page < jobCount; page++) {
                        // 采集原始数据
                        List<RemoteAlert> content = remoteAlertRepo.findAll(new PageRequest(page, size)).getContent();
                        List<Alert> partAlert = content
                                .parallelStream()
                                .map(RemoteAlert::toDo)
                                .collect(toList());
                        log.info("[pssp] 第{}/{}部分的历史数据[采集]完成,该部分处理了{}条数据", page + 1, jobCount, partAlert.size());

                        // 写入数据库
                        List<Alert> savedAlertList = alertRepo.save(partAlert);
                        log.info("[pssp] 第{}/{}部分的历史数据[写入DB]完成,该部分处理了{}条数据", page + 1, jobCount, savedAlertList.size());

                    }
                    log.info("[pssp] {} 月源报警历史数据抓取任务完成,size = {}", date, handledDataCount);
                } catch (Exception e) {
                    log.error("[pssp] [历史任务] 采集发生异常,目前采集月份为 {},采集数量为 {}, 异常栈 : {}", date, handledDataCount, e);
                } finally {
                    //历史任务完成之后重新设置回当前月份
                    log.info("[pssp] [历史任务] {} 月采集任务结束,将月份重新设置回{}时间", date, LocalDate.now());
                    CollectDataTask.setDynamicTableMonth(LocalDate.now());
                    //全部完成后再开启计划任务
                    CollectDataTask.trigger = true;
                }
            }
        });

    }

    /**
     * 获取所有更新的数据,步骤如下
     * <li>1.获取当前月份本地最新的一条报警数据id</li>
     * <li>2.向源数据库抓取在步骤1id之后的数据</li>
     * <li>3.数据入库</li>
     * <li>4.更新缓存相关数据</>
     */
    @Override
    public void fetchUpdatedData() {

        //从拦截器中获得当前应当处理的月份时间,并进行查询获得当前月份本地数据库报警数据的最新id
        String tableMonth = CollectDataTask.getDynamicTableMonth();
        Alert lastedAlert = alertRepo.findSpecifiedMonthLastedData(tableMonth);
        Long lastedDataId = Objects.isNull(lastedAlert) ? 0L : lastedAlert.getId();

        log.info("[pssp] 执行更新数据任务,本地数据库 {} 月最新数据的id为 {} ,向源数据库采集在这id之后的数据....", tableMonth, lastedDataId);
        log.debug("[pssp] [id_check],supposeId = {},actuallyId = {}", supposeId, lastedDataId);

        List<Alert> updatedData = remoteAlertRepo.findDataFromId(lastedDataId).stream()
                .map(RemoteAlert::toDo)
                .collect(toList());

        if (CollectionUtils.isEmpty(updatedData)) {
            log.info("[pssp] 本次更新采集没有要更新的数据");
        } else {
            log.info("[pssp] 采集完成,本次采集了 {} 条数据,数据写入本机数据库与Es数据库", updatedData.size());

            supposeId = updatedData.stream()
                    .map(Alert::getId)
                    .max(Comparator.naturalOrder())
                    .orElse(0L);
            //1.更新写入本地数据库
            List<Alert> savedUpdatedData = alertRepo.save(updatedData);
        }

    }

    /**
     * 查询所有数据
     */
    @Override
    public List<Alert> findAllData() {
        return alertRepo.findAll();
    }

    @Override
    public Page<Alert> findAllDataBetweenTime(LocalDate startTime, LocalDate endTime, Pageable pageable) {
        LocalDateTime maxEndTime = endTime.atTime(LocalTime.MAX);
        return alertRepo.findBetweenTime(
                        LocalToDate(startTime),
                        LocalToDate(maxEndTime),
                        pageable);
    }

    /**
     * 查询指定年月
     * 例如查询 2018年7月的数据 构造出2018-07-01 与 2018-08-00这样的两端字符串
     * 然后使用between查询即可,这样可以利用到数据库的索引
     */
    @Override
    public List<Alert> findAllDataByMonth(int year, int month) {
        LocalDate startMonth = LocalDate.of(year, month, 1);
        LocalDate nextMonth = startMonth.plusMonths(1);

        return alertRepo.findBetweenTime(LocalToDate(startMonth), LocalToDate(nextMonth));
    }

    /**
     * 查询指定年月日的告警数据
     */
    @Override
    public List<Alert> findAllDataByDate(int year, int month, int day) throws Exception {
        Date specDate = parseToDate(year, month, day);
        return alertRepo.findByTmFetch(specDate);
    }

    /**
     * 查询最近指定天数内的告警数据
     * 等同于查询 当前时间-指定天数时间 之后的所有数据
     */
    @Override
    public List<Alert> findRecentDaysData(int days) {
        LocalDate daysAgo = LocalDate.now()
                .minusDays(days);
        log.info("[pssp]查询日期在 {} 之后的数据", LocalToDate(daysAgo));
        return alertRepo.findAfterTime(LocalToDate(daysAgo));
    }

    /**
     * 查询最近指定天数内的告警数据
     *
     * @param specifiedTime 指定时间 精确到秒
     * @param days          指定最近几天的时间
     * @return 报警数据
     */
    @Override
    public List<Alert> findRecentDaysData(LocalDateTime specifiedTime, int days) {
        LocalDateTime daysAgo = specifiedTime
                .minusDays(days);
        log.info("[pssp]查询日期在 {} 之后的数据", LocalToDate(daysAgo));
        return alertRepo.findAfterTime(LocalToDate(daysAgo));
    }


    /**
     * 根据主键id集合查询alert对象
     *
     * @param primaryIds 主键id集合
     * @return 报警集合
     */
    @Override
    public List<Alert> findAll(List<Long> primaryIds) {
        return alertRepo.findAll(primaryIds);
    }


    @Override
    public Page<Alert> findAllAfterId(Long primaryId,Pageable pageable) {
        //check id
        Alert alert = alertRepo.findOne(primaryId);
        if (Objects.isNull(alert)) {
            throw new InvalidParameterException("提供的primaryId并不正确,请提供实际存在的primaryId");
        }
        return alertRepo.findByPrimaryIdAfter(primaryId, pageable);
    }


    /**
     * 查询出报警数据的最早时间点.
     *
     * @return 最早的一条报警数据的时间
     */
    @Override
    public Date findEarliestTime() {
        return alertRepo.findFirstByOrderByTmFetch().getTmFetch();
    }

    /*  以下为私有方法区域 */
}
