package com.zjty.fp.acq.union.config;

import com.alibaba.druid.pool.DruidDataSource;
import com.alibaba.druid.spring.boot.autoconfigure.DruidDataSourceBuilder;
import com.zjty.fp.acq.pssp.config.algorithm.AlertPreShardAlgo;
import com.zjty.fp.acq.pssp.config.algorithm.AlertRangeShardAlgo;
import io.shardingsphere.api.config.ShardingRuleConfiguration;
import io.shardingsphere.api.config.TableRuleConfiguration;
import io.shardingsphere.api.config.strategy.StandardShardingStrategyConfiguration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.env.Environment;

import javax.sql.DataSource;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import static io.shardingsphere.shardingjdbc.api.ShardingDataSourceFactory.createDataSource;

@ComponentScan
@Configuration
public class DataSourceConfig {

    @Autowired
    private Environment env;


    /**
     * fp-simc数据源，用于pssp平台采集数据
     */
    @Bean(name = "remoteDataSource")
    @Qualifier("remoteDataSource")
    @Primary
    @ConfigurationProperties(prefix = "spring.datasource.remote")
    public DataSource primaryDataSource() {
        return DruidDataSourceBuilder.create().build();
    }

    /**
     * fp-simc数据源，用于vomp平台采集数据
     */
    @Bean(name = "remoteDataSource2")
    @Qualifier("remoteDataSource2")
    @ConfigurationProperties(prefix = "spring.datasource.remote2")
    public DataSource vompDataSource() {
        return DruidDataSourceBuilder.create().build();
    }

    /**
     * fp-api数据源,用于正常提供api服务
     */
    @Bean(name = "locationDataSource")
    @Qualifier("locationDataSource")
    public DataSource secondaryDataSource() throws SQLException {

        ShardingRuleConfiguration shardingRuleConfig = new ShardingRuleConfiguration();
        shardingRuleConfig.getTableRuleConfigs().add(getAlertTableRuleConfiguration());
        shardingRuleConfig.getBindingTableGroups().add("pssp_alert");
        return createDataSource(
                createDataSourceMap(),
                shardingRuleConfig,
                new HashMap<>(),
                new Properties());
    }

    /**
     * fp-api数据源服务于{@link ##secondaryDataSource()}方法
     */
    private Map<String, DataSource> createDataSourceMap() throws SQLException {
        DruidDataSource dataSource = new DruidDataSource();
        dataSource.setDriverClassName(env.getProperty("spring.datasource.location.driver-class-name"));
        dataSource.setUrl(env.getProperty("spring.datasource.location.url"));
        dataSource.setUsername(env.getProperty("spring.datasource.location.username"));
        dataSource.setPassword(env.getProperty("spring.datasource.location.password"));

        /* 配置初始化大小、最小、最大 */
        dataSource.setDbType("mysql");
        dataSource.setInitialSize(4);
        dataSource.setMinIdle(4);
        dataSource.setMaxActive(20);

        /* 配置获取连接等待超时的时间 */
        dataSource.setMaxWait(60000);

        /* 配置间隔多久才进行一次检测，检测需要关闭的空闲连接，单位是毫秒 */
        dataSource.setTimeBetweenEvictionRunsMillis(60000);

        /* 配置一个连接在池中最小生存的时间，单位是毫秒 */
        dataSource.setMinEvictableIdleTimeMillis(300000);

        dataSource.setValidationQuery("SELECT 'x'");
        dataSource.setTestWhileIdle(true);
        dataSource.setTestOnBorrow(true);
        dataSource.setTestOnReturn(false);

        /* 打开PSCache，并且指定每个连接上PSCache的大小。
           如果用Oracle，则把poolPreparedStatements配置为true，
           mysql可以配置为false。分库分表较多的数据库，建议配置为false */
        dataSource.setPoolPreparedStatements(false);
        dataSource.setMaxPoolPreparedStatementPerConnectionSize(20);

        /* 配置监控统计拦截的filters */
        dataSource.setFilters("stat,wall,log4j");

        Map<String, DataSource> result = new HashMap<>();
        result.put("ds0", dataSource);
        return result;
    }

    /**
     * 设置表规则
     */
    private TableRuleConfiguration getAlertTableRuleConfiguration() {
        String actualDataNodes = "" +
                "ds0.pssp_alert_201712," +
                "ds0.pssp_alert_201807," +
                "ds0.pssp_alert_201808," +
                "ds0.pssp_alert_201809," +
                "ds0.pssp_alert_201810," +
                "ds0.pssp_alert_201811," +
                "ds0.pssp_alert_201812," +
                "ds0.pssp_alert_201901," +
                "ds0.pssp_alert_201902," +
                "ds0.pssp_alert_201903," +
                "ds0.pssp_alert_201904," +
                "ds0.pssp_alert_201905," +
                "ds0.pssp_alert_201906," +
                "ds0.pssp_alert_201907," +
                "ds0.pssp_alert_201908," +
                "ds0.pssp_alert_201909," +
                "ds0.pssp_alert_201910," +
                "ds0.pssp_alert_201911," +
                "ds0.pssp_alert_201912," +
                "ds0.pssp_alert_202001," +
                "ds0.pssp_alert_202002," +
                "ds0.pssp_alert_202003," +
                "ds0.pssp_alert_202004," +
                "ds0.pssp_alert_202005," +
                "ds0.pssp_alert_202006," +
                "ds0.pssp_alert_202007," +
                "ds0.pssp_alert_202008," +
                "ds0.pssp_alert_202009," +
                "ds0.pssp_alert_202010," +
                "ds0.pssp_alert_202011," +
                "ds0.pssp_alert_202012," +
                "ds0.pssp_alert_202101," +
                "ds0.pssp_alert_202102," +
                "ds0.pssp_alert_202103," +
                "ds0.pssp_alert_202104," +
                "ds0.pssp_alert_202105," +
                "ds0.pssp_alert_202106," +
                "ds0.pssp_alert_202107," +
                "ds0.pssp_alert_202108";
        TableRuleConfiguration result = new TableRuleConfiguration();
        result.setLogicTable("pssp_alert");
        result.setActualDataNodes(actualDataNodes);
        result.setKeyGeneratorColumnName("primary_id");
        result.setTableShardingStrategyConfig(new StandardShardingStrategyConfiguration(
                "tm_fetch",
                new AlertPreShardAlgo("pssp_alert_"),
                new AlertRangeShardAlgo("pssp_alert_")));
        return result;
    }
}
